using System;
using Oxide.Core;
using Oxide.Core.Libraries.Covalence;

namespace Oxide.Plugins
{
    [Info("ServerTitleUpdater", "Mevent", "1.0.0")]
    [Description("自动更新服务器标题日期")]
    public class ServerTitleUpdater : CovalencePlugin
    {
        private Configuration _config;
        private Timer _checkTimer;
        private DateTime _lastUpdate;

        private class Configuration
        {
            public string TitleTemplate { get; set; } = "『洋葱』{date}-3倍原倍硫磺-多人/无权/萌新入门";
            public string DateFormat { get; set; } = "M.dd日";
            public int UpdateHour { get; set; } = 0;
            public int UpdateMinute { get; set; } = 0;
            public int CheckIntervalSeconds { get; set; } = 60;
        }

        protected override void LoadConfig()
        {
            base.LoadConfig();
            try
            {
                _config = Config.ReadObject<Configuration>();
                if (_config == null) throw new Exception();
            }
            catch
            {
                LoadDefaultConfig();
            }
        }

        protected override void LoadDefaultConfig()
        {
            _config = new Configuration();
            SaveConfig();
        }

        protected override void SaveConfig() => Config.WriteObject(_config);

        private void Init()
        {
            _lastUpdate = DateTime.Now.Date.AddDays(-1);
        }

        private void OnServerInitialized()
        {
            UpdateTitle();
            StartTimer();
        }

        private void Unload()
        {
            _checkTimer?.Destroy();
        }

        private void StartTimer()
        {
            _checkTimer?.Destroy();
            _checkTimer = timer.Every(_config.CheckIntervalSeconds, CheckAndUpdate);
        }

        private void CheckAndUpdate()
        {
            var now = DateTime.Now;
            var targetTime = now.Date.AddHours(_config.UpdateHour).AddMinutes(_config.UpdateMinute);
            
            if (now >= targetTime && _lastUpdate < targetTime)
            {
                UpdateTitle();
                _lastUpdate = now;
            }
        }

        private void UpdateTitle()
        {
            var date = DateTime.Now.ToString(_config.DateFormat);
            var title = _config.TitleTemplate.Replace("{date}", date);
            
            server.Name = title;
            ConVar.Server.hostname = title;
            
            Puts($"标题已更新: {title}");
        }

        [Command("title.update")]
        private void CmdUpdateTitle(IPlayer player, string command, string[] args)
        {
            if (player.IsServer || player.HasPermission("servertitleupdater.admin"))
            {
                UpdateTitle();
                player.Reply("标题已手动更新");
            }
            else
            {
                player.Reply("无权限");
            }
        }

        [Command("title.set")]
        private void CmdSetTemplate(IPlayer player, string command, string[] args)
        {
            if (!player.IsServer && !player.HasPermission("servertitleupdater.admin"))
            {
                player.Reply("无权限");
                return;
            }

            if (args.Length == 0)
            {
                player.Reply($"当前模板: {_config.TitleTemplate}");
                player.Reply("用法: title.set <模板> (使用 {{date}} 作为日期占位符)");
                return;
            }

            _config.TitleTemplate = string.Join(" ", args);
            SaveConfig();
            UpdateTitle();
            player.Reply("模板已更新并应用");
        }

        [Command("title.time")]
        private void CmdSetTime(IPlayer player, string command, string[] args)
        {
            if (!player.IsServer && !player.HasPermission("servertitleupdater.admin"))
            {
                player.Reply("无权限");
                return;
            }

            if (args.Length != 2 || !int.TryParse(args[0], out var hour) || !int.TryParse(args[1], out var minute))
            {
                player.Reply($"当前更新时间: {_config.UpdateHour:D2}:{_config.UpdateMinute:D2}");
                player.Reply("用法: title.time <小时> <分钟>");
                player.Reply("例如: title.time 12 0");
                return;
            }

            if (hour < 0 || hour > 23 || minute < 0 || minute > 59)
            {
                player.Reply("时间格式错误");
                return;
            }

            _config.UpdateHour = hour;
            _config.UpdateMinute = minute;
            SaveConfig();
            player.Reply($"更新时间已设置为: {hour:D2}:{minute:D2}");
        }

        [Command("title.info")]
        private void CmdInfo(IPlayer player, string command, string[] args)
        {
            if (!player.IsServer && !player.HasPermission("servertitleupdater.admin"))
            {
                player.Reply("无权限");
                return;
            }

            var now = DateTime.Now;
            var nextUpdate = now.Date.AddHours(_config.UpdateHour).AddMinutes(_config.UpdateMinute);
            if (nextUpdate < now) nextUpdate = nextUpdate.AddDays(1);

            player.Reply("=== 服务器标题更新器 ===");
            player.Reply($"当前标题: {server.Name}");
            player.Reply($"标题模板: {_config.TitleTemplate}");
            player.Reply($"日期格式: {_config.DateFormat}");
            player.Reply($"更新时间: 每天 {_config.UpdateHour:D2}:{_config.UpdateMinute:D2}");
            player.Reply($"下次更新: {nextUpdate:yyyy-MM-dd HH:mm:ss}");
            player.Reply($"上次更新: {_lastUpdate:yyyy-MM-dd HH:mm:ss}");
        }
    }
}
