<?php
/**
 * 朋友圈访客查询系统
 * 
 * @author    锅岛主
 * @link      https://ggdao.net
 * @wechat    IJG55555
 * @copyright Copyright (c) 2026 锅岛主 (https://ggdao.net)
 * @license   保留所有权利
 */

require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Admin.php';

// 验证管理员权限
$admin = new Admin();
$admin->checkAccess();

// 初始化数据库连接
$db = Database::getInstance();

// 获取筛选参数
$status = isset($_GET['status']) ? $_GET['status'] : '';
$start_date = isset($_GET['start_date']) ? $_GET['start_date'] : '';
$end_date = isset($_GET['end_date']) ? $_GET['end_date'] : '';
$search = isset($_GET['search']) ? $_GET['search'] : '';
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$per_page = 20;

// 构建查询条件
$where = [];
$params = [];

if ($status) {
    $where[] = "vo.status = ?";
    $params[] = $status;
}

if ($start_date) {
    $where[] = "vo.created_at >= ?";
    $params[] = $start_date . ' 00:00:00';
}

if ($end_date) {
    $where[] = "vo.created_at <= ?";
    $params[] = $end_date . ' 23:59:59';
}

if ($search) {
    $where[] = "(vo.order_no LIKE ? OR u.nickname LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

$where_clause = $where ? "WHERE " . implode(" AND ", $where) : "";

// 获取总记录数
$count_sql = "SELECT COUNT(*) as total FROM vip_orders vo 
              LEFT JOIN users u ON vo.user_id = u.id 
              $where_clause";
$stmt = $db->query($count_sql, $params);
$total = $stmt->fetch()['total'];
$total_pages = ceil($total / $per_page);

// 获取订单列表
$offset = ($page - 1) * $per_page;
$sql = "SELECT vo.*, u.nickname, u.avatar 
        FROM vip_orders vo 
        LEFT JOIN users u ON vo.user_id = u.id 
        $where_clause 
        ORDER BY vo.created_at DESC 
        LIMIT ? OFFSET ?";
$params[] = $per_page;
$params[] = $offset;
$stmt = $db->query($sql, $params);
$orders = array_reverse($stmt->fetchAll());

// 获取统计数据
$stats_sql = "SELECT 
              COUNT(*) as total_orders,
              SUM(CASE WHEN status = 'paid' THEN 1 ELSE 0 END) as paid_orders,
              SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending_orders,
              SUM(CASE WHEN status = 'cancelled' THEN 1 ELSE 0 END) as cancelled_orders,
              SUM(CASE WHEN status = 'paid' THEN amount ELSE 0 END) as total_amount
              FROM vip_orders vo 
              $where_clause";
$stmt = $db->query($stats_sql, $params);
$stats = $stmt->fetch();

// 页面标题
$page_title = "订单管理";
$page_icon = "fas fa-shopping-cart";
require_once 'layout/header.php';
?>

<style>
/* 重置基础样式 */
:root {
    --primary-color: #4f46e5;
    --primary-hover: #4338ca;
    --success-color: #10b981;
    --warning-color: #f59e0b;
    --danger-color: #ef4444;
    --text-primary: #1f2937;
    --text-secondary: #6b7280;
    --bg-light: #f9fafb;
    --bg-white: #ffffff;
    --border-color: #e5e7eb;
    --border-radius: 8px;
    --shadow-sm: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
    --shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
}

/* 整体布局 */
.content-wrapper {
    background: var(--bg-light);
    padding: 2rem;
    min-height: 100vh;
}

.page-header {
    margin-bottom: 2rem;
}

.page-header h1 {
    font-size: 1.875rem;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 0.5rem;
}

.breadcrumb {
    color: var(--text-secondary);
    font-size: 0.875rem;
}

.breadcrumb a {
    color: var(--primary-color);
    text-decoration: none;
}

.breadcrumb a:hover {
    text-decoration: underline;
}

/* 统计卡片样式 */
.stats-container {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.stat-card {
    background: var(--bg-white);
    border-radius: var(--border-radius);
    padding: 1.5rem;
    box-shadow: var(--shadow);
    transition: all 0.3s ease;
    border: 1px solid var(--border-color);
}

.stat-card:hover {
    transform: translateY(-4px);
    box-shadow: var(--shadow-lg);
}

.stat-card .stat-value {
    font-size: 2.25rem;
    font-weight: 700;
    color: var(--primary-color);
    margin-bottom: 0.75rem;
    line-height: 1;
}

.stat-card .stat-label {
    color: var(--text-secondary);
    font-size: 0.875rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.stat-card .stat-label i {
    font-size: 1.25rem;
    opacity: 0.8;
}

/* 搜索区域样式 */
.search-section {
    background: var(--bg-white);
    border-radius: var(--border-radius);
    padding: 1.5rem;
    margin-bottom: 2rem;
    box-shadow: var(--shadow);
    border: 1px solid var(--border-color);
}

.search-form {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(240px, 1fr));
    gap: 1rem;
}

.form-group {
    position: relative;
}

.form-control {
    width: 100%;
    padding: 0.75rem 1rem;
    border: 1px solid var(--border-color);
    border-radius: var(--border-radius);
    font-size: 0.875rem;
    transition: all 0.2s ease;
    background-color: var(--bg-white);
}

.form-control:focus {
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
    outline: none;
}

.input-group {
    display: flex;
    align-items: center;
}

.input-group-text {
    padding: 0.75rem 1rem;
    background: var(--bg-light);
    border: 1px solid var(--border-color);
    border-right: none;
    border-radius: var(--border-radius) 0 0 var(--border-radius);
    color: var(--text-secondary);
}

.input-group .form-control {
    border-radius: 0 var(--border-radius) var(--border-radius) 0;
}

/* 订单列表样式 */
.orders-section {
    background: var(--bg-white);
    border-radius: var(--border-radius);
    box-shadow: var(--shadow);
    border: 1px solid var(--border-color);
    overflow: hidden;
}

.orders-header {
    padding: 1.5rem;
    border-bottom: 1px solid var(--border-color);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.orders-header h2 {
    font-size: 1.25rem;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0;
}

.table {
    width: 100%;
    border-collapse: separate;
    border-spacing: 0;
}

.table th {
    background: var(--bg-light);
    padding: 1rem 1.5rem;
    font-weight: 600;
    color: var(--text-primary);
    text-align: left;
    font-size: 0.875rem;
    border-bottom: 1px solid var(--border-color);
}

.table td {
    padding: 1rem 1.5rem;
    border-bottom: 1px solid var(--border-color);
    color: var(--text-secondary);
    font-size: 0.875rem;
}

.table tr:hover td {
    background: var(--bg-light);
}

/* 状态标签样式 */
.badge {
    padding: 0.5rem 0.75rem;
    border-radius: 9999px;
    font-size: 0.75rem;
    font-weight: 500;
    display: inline-flex;
    align-items: center;
    gap: 0.375rem;
    min-width: 80px;
    justify-content: center;
}

.badge-warning {
    background: var(--warning-color);
    color: white;
}

.badge-success {
    background: var(--success-color);
    color: white;
}

.badge-danger {
    background: var(--danger-color);
    color: white;
}

.badge-info {
    background: var(--primary-color);
    color: white;
}

/* 按钮样式 */
.btn {
    padding: 0.75rem 1.5rem;
    border-radius: var(--border-radius);
    font-weight: 500;
    font-size: 0.875rem;
    transition: all 0.2s ease;
    border: none;
    cursor: pointer;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
}

.btn-primary {
    background: var(--primary-color);
    color: white;
}

.btn-primary:hover {
    background: var(--primary-hover);
    transform: translateY(-1px);
}

.btn-secondary {
    background: var(--bg-light);
    color: var(--text-primary);
    border: 1px solid var(--border-color);
}

.btn-secondary:hover {
    background: var(--border-color);
}

.btn-sm {
    padding: 0.5rem 1rem;
    font-size: 0.75rem;
}

/* 分页样式 */
.pagination {
    display: flex;
    gap: 0.5rem;
    padding: 1.5rem;
    justify-content: flex-end;
}

.page-link {
    padding: 0.5rem 1rem;
    border-radius: var(--border-radius);
    background: var(--bg-light);
    color: var(--text-primary);
    text-decoration: none;
    transition: all 0.2s ease;
    font-size: 0.875rem;
    border: 1px solid var(--border-color);
}

.page-link:hover {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
}

.page-item.active .page-link {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
}

/* 模态框样式 */
.modal-content {
    border-radius: var(--border-radius);
    border: none;
    box-shadow: var(--shadow-lg);
}

.modal-header {
    background: var(--bg-light);
    border-radius: var(--border-radius) var(--border-radius) 0 0;
    padding: 1.5rem;
    border-bottom: 1px solid var(--border-color);
}

.modal-title {
    font-size: 1.25rem;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0;
}

.modal-body {
    padding: 2rem;
}

/* 响应式优化 */
@media (max-width: 768px) {
    .content-wrapper {
        padding: 1rem;
    }
    
    .search-form {
        grid-template-columns: 1fr;
    }
    
    .table-responsive {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
    }
    
    .orders-header {
        flex-direction: column;
        gap: 1rem;
        align-items: stretch;
    }
    
    .orders-header h2 {
        text-align: center;
    }
    
    .pagination {
        justify-content: center;
    }
}

/* 动画效果 */
@keyframes fadeIn {
    from {
        opacity: 0;
        transform: translateY(10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.fade-in {
    animation: fadeIn 0.5s ease forwards;
}

/* 用户头像样式 */
.img-circle {
    border-radius: 50%;
    object-fit: cover;
}

/* 金额样式 */
.text-success {
    color: var(--success-color) !important;
    font-weight: 600;
}

/* 表格内用户信息样式 */
.user-info {
    display: flex;
    align-items: center;
    gap: 1rem;
    padding: 0.5rem 0;
}

.user-info img {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    object-fit: cover;
    border: 2px solid var(--border-color);
}

.user-info span {
    color: var(--text-primary);
    font-weight: 500;
    font-size: 0.9rem;
}
</style>

<div class="content-wrapper">
    <div class="page-header">
        <h1>订单管理</h1>
        <nav class="breadcrumb">
            <a href="/admin/index.php">首页</a> / 
            <span>订单管理</span>
        </nav>
    </div>

    <!-- 统计卡片 -->
    <div class="stats-container">
        <div class="stat-card">
            <div class="stat-value"><?php echo $stats['total_orders']; ?></div>
            <div class="stat-label">
                <i class="fas fa-shopping-cart"></i>
                总订单数
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-value"><?php echo $stats['paid_orders']; ?></div>
            <div class="stat-label">
                <i class="fas fa-check-circle"></i>
                已支付订单
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-value"><?php echo $stats['pending_orders']; ?></div>
            <div class="stat-label">
                <i class="fas fa-clock"></i>
                待支付订单
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-value">￥<?php echo number_format($stats['total_amount'], 2); ?></div>
            <div class="stat-label">
                <i class="fas fa-yen-sign"></i>
                总交易金额
            </div>
        </div>
    </div>

    <!-- 搜索区域 -->
    <div class="search-section">
        <form class="search-form">
            <div class="form-group">
                <select name="status" class="form-control">
                    <option value="">全部状态</option>
                    <option value="pending" <?php echo $status == 'pending' ? 'selected' : ''; ?>>待支付</option>
                    <option value="paid" <?php echo $status == 'paid' ? 'selected' : ''; ?>>已支付</option>
                    <option value="cancelled" <?php echo $status == 'cancelled' ? 'selected' : ''; ?>>已取消</option>
                </select>
            </div>
            <div class="form-group">
                <div class="input-group">
                    <div class="input-group-prepend">
                        <span class="input-group-text">
                            <i class="far fa-calendar-alt"></i>
                        </span>
                    </div>
                    <input type="date" name="start_date" class="form-control" value="<?php echo $start_date; ?>" placeholder="开始日期">
                </div>
            </div>
            <div class="form-group">
                <div class="input-group">
                    <div class="input-group-prepend">
                        <span class="input-group-text">
                            <i class="far fa-calendar-alt"></i>
                        </span>
                    </div>
                    <input type="date" name="end_date" class="form-control" value="<?php echo $end_date; ?>" placeholder="结束日期">
                </div>
            </div>
            <div class="form-group">
                <div class="input-group">
                    <div class="input-group-prepend">
                        <span class="input-group-text">
                            <i class="fas fa-search"></i>
                        </span>
                    </div>
                    <input type="text" name="search" class="form-control" value="<?php echo htmlspecialchars($search); ?>" placeholder="订单号/用户昵称">
                </div>
            </div>
            <button type="submit" class="btn btn-primary">
                <i class="fas fa-search mr-1"></i> 搜索
            </button>
            <a href="orders.php" class="btn btn-secondary">
                <i class="fas fa-redo mr-1"></i> 重置
            </a>
        </form>
    </div>

    <!-- 订单列表 -->
    <div class="orders-section">
        <div class="orders-header">
            <h2>订单列表</h2>
            <button class="btn btn-primary" onclick="exportOrders()">
                <i class="fas fa-download"></i> 导出数据
            </button>
        </div>
        <div class="table-responsive">
            <table class="table table-hover text-nowrap">
                <thead>
                    <tr>
                        <th>订单号</th>
                        <th>用户</th>
                        <th>金额</th>
                        <th>VIP天数</th>
                        <th>状态</th>
                        <th>创建时间</th>
                        <th>支付时间</th>
                        <th>操作</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($orders as $order): ?>
                    <tr>
                        <td>
                            <span class="text-muted"><?php echo $order['order_no']; ?></span>
                        </td>
                        <td>
                            <div class="user-info">
                                <img src="<?php echo $order['avatar']; ?>" alt="用户头像">
                                <span><?php echo htmlspecialchars($order['nickname']); ?></span>
                            </div>
                        </td>
                        <td>
                            <span class="text-success font-weight-bold">
                                ￥<?php echo number_format($order['amount'], 2); ?>
                            </span>
                        </td>
                        <td>
                            <span class="badge badge-info">
                                <?php echo $order['days']; ?>天
                            </span>
                        </td>
                        <td>
                            <?php
                            $status_class = [
                                'pending' => 'warning',
                                'paid' => 'success',
                                'cancelled' => 'danger'
                            ];
                            $status_text = [
                                'pending' => '未支付',
                                'paid' => '已支付',
                                'cancelled' => '已取消'
                            ];
                            ?>
                            <span class="badge badge-<?php echo $status_class[$order['status']]; ?>">
                                <?php echo $status_text[$order['status']]; ?>
                            </span>
                        </td>
                        <td>
                            <span class="text-muted">
                                <?php echo date('Y-m-d H:i:s', strtotime($order['created_at'])); ?>
                            </span>
                        </td>
                        <td>
                            <?php if ($order['paid_at']): ?>
                                <span class="text-success">
                                    <?php echo date('Y-m-d H:i:s', strtotime($order['paid_at'])); ?>
                                </span>
                            <?php else: ?>
                                <span class="text-muted">-</span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <button type="button" class="btn btn-info btn-sm" onclick="viewOrder(<?php echo $order['id']; ?>)">
                                <i class="fas fa-eye"></i> 查看
                            </button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <div class="card-footer clearfix">
            <?php if ($total_pages > 1): ?>
            <ul class="pagination pagination-sm m-0 float-right">
                <?php if ($page > 1): ?>
                <li class="page-item">
                    <a class="page-link" href="?page=<?php echo $page-1; ?>&status=<?php echo $status; ?>&start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>&search=<?php echo urlencode($search); ?>">
                        <i class="fas fa-chevron-left"></i>
                    </a>
                </li>
                <?php endif; ?>
                
                <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                    <a class="page-link" href="?page=<?php echo $i; ?>&status=<?php echo $status; ?>&start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>&search=<?php echo urlencode($search); ?>">
                        <?php echo $i; ?>
                    </a>
                </li>
                <?php endfor; ?>
                
                <?php if ($page < $total_pages): ?>
                <li class="page-item">
                    <a class="page-link" href="?page=<?php echo $page+1; ?>&status=<?php echo $status; ?>&start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>&search=<?php echo urlencode($search); ?>">
                        <i class="fas fa-chevron-right"></i>
                    </a>
                </li>
                <?php endif; ?>
            </ul>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- 订单详情模态框 -->
<div class="modal fade" id="orderModal" tabindex="-1" role="dialog" aria-labelledby="orderModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="orderModalLabel">
                    <i class="fas fa-info-circle mr-1"></i>
                    订单详情
                </h5>
                <button type="button" class="close" data-bs-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <div id="orderDetails"></div>
            </div>
        </div>
    </div>
</div>

<script>
$(function() {
    // 初始化Select2
    $('.select2').select2({
        theme: 'bootstrap4',
        width: '100%'
    });
    
    // 初始化日期选择器
    $('input[type="date"]').daterangepicker({
        singleDatePicker: true,
        showDropdowns: true,
        locale: {
            format: 'YYYY-MM-DD'
        },
        autoUpdateInput: false
    });
    
    // 添加表格行动画
    $('.table tbody tr').each(function(index) {
        $(this).css('animation', `fadeIn 0.5s ease ${index * 0.1}s`);
    });
    
    // 添加卡片动画
    $('.small-box').each(function(index) {
        $(this).css('animation', `fadeIn 0.5s ease ${index * 0.1}s`);
    });
});

function viewOrder(id) {
    $.get('api/get_order.php', { id: id }, function(response) {
        if (response.success) {
            var order = response.data;
            var html = `
                <div class="table-responsive">
                    <table class="table table-bordered">
                        <tr>
                            <th style="width: 150px;">订单号：</th>
                            <td>${order.order_no}</td>
                        </tr>
                        <tr>
                            <th>用户：</th>
                            <td>
                                <div class="d-flex align-items-center">
                                    <img src="${order.avatar}" class="img-circle mr-2" width="32" height="32">
                                    ${order.nickname}
                                </div>
                            </td>
                        </tr>
                        <tr>
                            <th>金额：</th>
                            <td>
                                <span class="text-success font-weight-bold">
                                    ￥${parseFloat(order.amount).toFixed(2)}
                                </span>
                            </td>
                        </tr>
                        <tr>
                            <th>VIP天数：</th>
                            <td>
                                <span class="badge badge-info">
                                    ${order.days}天
                                </span>
                            </td>
                        </tr>
                        <tr>
                            <th>状态：</th>
                            <td>
                                <span class="badge badge-${getStatusClass(order.status)}">
                                    ${getStatusText(order.status)}
                                </span>
                            </td>
                        </tr>
                        <tr>
                            <th>创建时间：</th>
                            <td>${new Date(order.created_at).toLocaleString()}</td>
                        </tr>
                        <tr>
                            <th>支付时间：</th>
                            <td>${order.paid_at ? new Date(order.paid_at).toLocaleString() : '-'}</td>
                        </tr>
                        <tr>
                            <th>交易号：</th>
                            <td>${order.trade_no || '-'}</td>
                        </tr>
                    </table>
                </div>
            `;
            $('#orderDetails').html(html);
            $('#orderModal').modal('show');
        } else {
            alert(response.message || '获取订单详情失败');
        }
    });
}

function exportOrders() {
    var params = new URLSearchParams(window.location.search);
    window.location.href = 'api/export_orders.php?' + params.toString();
}

function getStatusClass(status) {
    var classes = {
        'pending': 'warning',
        'paid': 'success',
        'cancelled': 'danger'
    };
    return classes[status] || 'secondary';
}

function getStatusText(status) {
    var texts = {
        'pending': '未支付',
        'paid': '已支付',
        'cancelled': '已取消'
    };
    return texts[status] || status;
}
</script>

<?php require_once 'layout/footer.php'; ?> 