<?php
/**
 * 朋友圈访客查询系统
 * 
 * @author    锅岛主
 * @link      https://ggdao.net
 * @wechat    IJG55555
 * @copyright Copyright (c) 2026 锅岛主 (https://ggdao.net)
 * @license   保留所有权利
 */

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../includes/Database.php';
require_once __DIR__ . '/../includes/Admin.php';
require_once __DIR__ . '/../includes/VipCardService.php';

// 验证管理员权限
$admin = new Admin();
$admin->checkAccess();

$db = Database::getInstance();
$cardService = new VipCardService();

// 处理卡密生成
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'generate') {
        $count = intval($_POST['count'] ?? 0);
        $days = intval($_POST['days'] ?? 0);
        $expireTime = !empty($_POST['expire_time']) ? $_POST['expire_time'] : null;
        
        if ($count > 0 && $days > 0) {
            $result = $cardService->generateCards($count, $days, $expireTime);
            if ($result['code'] === 0) {
                $message = ['type' => 'success', 'text' => $result['msg']];
                // 将生成的卡密保存到session，用于导出
                $_SESSION['generated_cards'] = $result['data']['cards'];
            } else {
                $message = ['type' => 'danger', 'text' => $result['msg']];
            }
        } else {
            $message = ['type' => 'danger', 'text' => '请输入正确的生成数量和天数'];
        }
    } elseif ($_POST['action'] === 'invalidate' && !empty($_POST['card_nos'])) {
        $result = $cardService->invalidateCards($_POST['card_nos']);
        if ($result['code'] === 0) {
            $message = ['type' => 'success', 'text' => $result['msg']];
        } else {
            $message = ['type' => 'danger', 'text' => $result['msg']];
        }
    }
}

// 获取卡密统计信息
$stats = $cardService->getCardStats();

// 分页参数
$page = max(1, intval($_GET['page'] ?? 1));
$limit = 20;
$offset = ($page - 1) * $limit;

// 搜索条件
$where = "1=1";
$params = [];

if (!empty($_GET['status'])) {
    $where .= " AND status = ?";
    $params[] = $_GET['status'];
}

if (!empty($_GET['keyword'])) {
    $where .= " AND (card_no LIKE ? OR batch_no LIKE ?)";
    $params[] = "%{$_GET['keyword']}%";
    $params[] = "%{$_GET['keyword']}%";
}

// 获取卡密列表
$sql = "SELECT c.*, u.nickname as user_nickname 
        FROM vip_cards c 
        LEFT JOIN users u ON c.used_user_id = u.id 
        WHERE {$where} 
        ORDER BY c.id DESC 
        LIMIT {$offset}, {$limit}";
$cards = $db->query($sql, $params)->fetchAll();

// 获取总数
$sql = "SELECT COUNT(*) as total FROM vip_cards WHERE {$where}";
$total = $db->query($sql, $params)->fetch()['total'];
$totalPages = ceil($total / $limit);

// 设置页面标题
$pageTitle = '卡密管理';
$pageIcon = 'fas fa-key';

// 加载页头
require_once __DIR__ . '/layout/header.php';
?>

<div class="container-fluid">
    <?php if (isset($message)): ?>
    <div class="alert alert-<?php echo $message['type']; ?> alert-dismissible fade show">
        <?php echo $message['text']; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
    <?php endif; ?>

    <!-- 统计信息 -->
    <div class="row mb-4">
        <div class="col-md-3">
            <div class="card bg-primary text-white">
                <div class="card-body">
                    <h6 class="card-title">总卡密数</h6>
                    <h3 class="mb-0"><?php echo number_format($stats['total']); ?></h3>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card bg-success text-white">
                <div class="card-body">
                    <h6 class="card-title">未使用</h6>
                    <h3 class="mb-0"><?php echo number_format($stats['unused']); ?></h3>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card bg-info text-white">
                <div class="card-body">
                    <h6 class="card-title">已使用</h6>
                    <h3 class="mb-0"><?php echo number_format($stats['used']); ?></h3>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card bg-danger text-white">
                <div class="card-body">
                    <h6 class="card-title">已作废</h6>
                    <h3 class="mb-0"><?php echo number_format($stats['invalid']); ?></h3>
                </div>
            </div>
        </div>
    </div>

    <div class="row mb-4">
        <!-- 生成卡密 -->
        <div class="col-md-6">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">生成卡密</h5>
                </div>
                <div class="card-body">
                    <form method="post" action="">
                        <input type="hidden" name="action" value="generate">
                        <div class="row">
                            <div class="col-md-4">
                                <div class="form-group mb-3">
                                    <label for="count">生成数量</label>
                                    <input type="number" class="form-control" id="count" name="count" 
                                        min="1" max="100" value="10" required>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="form-group mb-3">
                                    <label for="days">VIP天数</label>
                                    <input type="number" class="form-control" id="days" name="days" 
                                        min="1" value="30" required>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="form-group mb-3">
                                    <label for="expire_time">过期时间</label>
                                    <input type="date" class="form-control" id="expire_time" name="expire_time"
                                        min="<?php echo date('Y-m-d'); ?>">
                                </div>
                            </div>
                        </div>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-plus me-1"></i>生成卡密
                        </button>
                        <?php if (!empty($_SESSION['generated_cards'])): ?>
                        <a href="export_cards.php" class="btn btn-success">
                            <i class="fas fa-download me-1"></i>导出最近生成
                        </a>
                        <?php endif; ?>
                    </form>
                </div>
            </div>
        </div>

        <!-- 搜索框 -->
        <div class="col-md-6">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">搜索卡密</h5>
                </div>
                <div class="card-body">
                    <form method="get" class="mb-0">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="keyword">关键词</label>
                                    <input type="text" class="form-control" id="keyword" name="keyword" 
                                        value="<?php echo htmlspecialchars($_GET['keyword'] ?? ''); ?>" 
                                        placeholder="卡密号码/批次号">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="status">状态</label>
                                    <select class="form-control" id="status" name="status">
                                        <option value="">全部</option>
                                        <option value="unused" <?php echo ($_GET['status'] ?? '') === 'unused' ? 'selected' : ''; ?>>未使用</option>
                                        <option value="used" <?php echo ($_GET['status'] ?? '') === 'used' ? 'selected' : ''; ?>>已使用</option>
                                        <option value="invalid" <?php echo ($_GET['status'] ?? '') === 'invalid' ? 'selected' : ''; ?>>已作废</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-search me-1"></i>搜索
                        </button>
                        <a href="?" class="btn btn-secondary">重置</a>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- 卡密列表 -->
    <div class="card">
        <div class="card-header">
            <h5 class="mb-0">卡密列表</h5>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-striped">
                    <thead>
                        <tr>
                            <th>卡密号码</th>
                            <th>VIP天数</th>
                            <th>批次号</th>
                            <th>状态</th>
                            <th>使用者</th>
                            <th>使用时间</th>
                            <th>过期时间</th>
                            <th>操作</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($cards as $card): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($card['card_no']); ?></td>
                            <td><?php echo $card['days']; ?>天</td>
                            <td><?php echo htmlspecialchars($card['batch_no']); ?></td>
                            <td>
                                <?php
                                $statusClass = [
                                    'unused' => 'success',
                                    'used' => 'info',
                                    'invalid' => 'danger'
                                ];
                                $statusText = [
                                    'unused' => '未使用',
                                    'used' => '已使用',
                                    'invalid' => '已作废'
                                ];
                                ?>
                                <span class="badge bg-<?php echo $statusClass[$card['status']]; ?>">
                                    <?php echo $statusText[$card['status']]; ?>
                                </span>
                            </td>
                            <td><?php echo $card['user_nickname'] ? htmlspecialchars($card['user_nickname']) : '-'; ?></td>
                            <td><?php echo $card['used_time'] ? date('Y-m-d H:i', strtotime($card['used_time'])) : '-'; ?></td>
                            <td><?php echo $card['expire_time'] ? date('Y-m-d', strtotime($card['expire_time'])) : '-'; ?></td>
                            <td>
                                <?php if ($card['status'] === 'unused'): ?>
                                <form method="post" style="display: inline-block;" onsubmit="return confirm('确定要作废这张卡密吗？');">
                                    <input type="hidden" name="action" value="invalidate">
                                    <input type="hidden" name="card_nos[]" value="<?php echo htmlspecialchars($card['card_no']); ?>">
                                    <button type="submit" class="btn btn-sm btn-danger">
                                        <i class="fas fa-times"></i> 作废
                                    </button>
                                </form>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        <?php if (empty($cards)): ?>
                        <tr>
                            <td colspan="8" class="text-center py-3">暂无数据</td>
                        </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <!-- 分页 -->
            <?php if ($totalPages > 1): ?>
            <nav class="mt-4">
                <ul class="pagination justify-content-center">
                    <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                    <li class="page-item <?php echo $page === $i ? 'active' : ''; ?>">
                        <a class="page-link" href="?page=<?php echo $i; ?>&keyword=<?php echo urlencode($_GET['keyword'] ?? ''); ?>&status=<?php echo urlencode($_GET['status'] ?? ''); ?>">
                            <?php echo $i; ?>
                        </a>
                    </li>
                    <?php endfor; ?>
                </ul>
            </nav>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php require_once __DIR__ . '/layout/footer.php'; ?> 