<?php
/**
 * 朋友圈访客查询系统
 * 
 * @author    锅岛主
 * @link      https://ggdao.net
 * @wechat    IJG55555
 * @copyright Copyright (c) 2026 锅岛主 (https://ggdao.net)
 * @license   保留所有权利
 */

require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Admin.php';

// 检查管理员是否登录
if (!isset($_SESSION['admin'])) {
    header('Location: /admin/login.php');
    exit;
}

$db = Database::getInstance();

// 处理添加/编辑套餐
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $id = $_POST['id'] ?? null;
    $data = [
        'name' => $_POST['name'],
        'days' => (int)$_POST['days'],
        'price' => (float)$_POST['price'],
        'description' => $_POST['description'],
        'status' => $_POST['status'],
        'sort_order' => (int)$_POST['sort_order']
    ];
    
    if ($id) {
        // 更新套餐
        $db->update('vip_plans', $data, 'id = ?', [$id]);
        $_SESSION['success'] = '套餐更新成功！';
    } else {
        // 添加套餐
        $db->insert('vip_plans', $data);
        $_SESSION['success'] = '套餐添加成功！';
    }
    
    header('Location: /admin/vip_plans.php');
    exit;
}

// 处理删除套餐
if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['id'])) {
    $db->delete('vip_plans', 'id = ?', [$_GET['id']]);
    $_SESSION['success'] = '套餐删除成功！';
    header('Location: /admin/vip_plans.php');
    exit;
}

// 获取所有套餐
$plans = $db->query("SELECT * FROM vip_plans ORDER BY sort_order ASC")->fetchAll();

// 加载管理后台模板
require_once 'layout/header.php';
?>

<div class="content-wrapper">
    <div class="content-header">
        <div class="container-fluid">
            <div class="row mb-2">
                <div class="col-sm-6">
                    <h1 class="m-0">VIP套餐管理</h1>
                </div>
            </div>
        </div>
    </div>

    <section class="content">
        <div class="container-fluid">
            <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?php 
                echo $_SESSION['success'];
                unset($_SESSION['success']);
                ?>
                <button type="button" class="close" data-bs-dismiss="alert">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <?php endif; ?>

            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">VIP套餐列表</h3>
                    <div class="card-tools">
                        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#planModal">
                            添加套餐
                        </button>
                    </div>
                </div>
                <div class="card-body">
                    <table class="table table-bordered">
                        <thead>
                            <tr>
                                <th style="width: 60px">排序</th>
                                <th>套餐名称</th>
                                <th>天数</th>
                                <th>价格</th>
                                <th>描述</th>
                                <th>状态</th>
                                <th style="width: 150px">操作</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($plans as $plan): ?>
                            <tr>
                                <td><?php echo $plan['sort_order']; ?></td>
                                <td><?php echo htmlspecialchars($plan['name']); ?></td>
                                <td><?php echo $plan['days']; ?>天</td>
                                <td>¥<?php echo number_format($plan['price'], 2); ?></td>
                                <td><?php echo htmlspecialchars($plan['description']); ?></td>
                                <td>
                                    <span class="badge badge-<?php echo $plan['status'] === 'active' ? 'success' : 'secondary'; ?>">
                                        <?php echo $plan['status'] === 'active' ? '启用' : '禁用'; ?>
                                    </span>
                                </td>
                                <td>
                                    <button type="button" 
                                            class="btn btn-sm btn-info edit-plan" 
                                            data-plan='<?php echo json_encode($plan); ?>'>
                                        编辑
                                    </button>
                                    <a href="?action=delete&id=<?php echo $plan['id']; ?>" 
                                       class="btn btn-sm btn-danger"
                                       onclick="return confirm('确定要删除这个套餐吗？')">
                                        删除
                                    </a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </section>
</div>

<!-- 添加/编辑套餐模态框 -->
<div class="modal fade" id="planModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form action="" method="post">
                <input type="hidden" name="id" id="planId">
                <div class="modal-header">
                    <h5 class="modal-title">添加/编辑套餐</h5>
                    <button type="button" class="close" data-bs-dismiss="modal">
                        <span>&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <div class="form-group">
                        <label>套餐名称</label>
                        <input type="text" class="form-control" name="name" required>
                    </div>
                    <div class="form-group">
                        <label>天数</label>
                        <input type="number" class="form-control" name="days" required min="1">
                    </div>
                    <div class="form-group">
                        <label>价格</label>
                        <input type="number" class="form-control" name="price" required min="0" step="0.01">
                    </div>
                    <div class="form-group">
                        <label>描述</label>
                        <input type="text" class="form-control" name="description">
                    </div>
                    <div class="form-group">
                        <label>排序</label>
                        <input type="number" class="form-control" name="sort_order" required min="0" value="0">
                    </div>
                    <div class="form-group">
                        <label>状态</label>
                        <select class="form-control" name="status">
                            <option value="active">启用</option>
                            <option value="inactive">禁用</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">取消</button>
                    <button type="submit" class="btn btn-primary">保存</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// 确保在DOM和jQuery加载完成后执行
document.addEventListener('DOMContentLoaded', function() {
    // 检查jQuery是否加载
    if (typeof jQuery === 'undefined') {
        console.error('jQuery未加载，请检查footer.php中的jQuery引入');
        return;
    }
    
    // 编辑套餐
    document.querySelectorAll('.edit-plan').forEach(button => {
        button.addEventListener('click', function() {
            const plan = JSON.parse(this.dataset.plan);
            const modal = $('#planModal');
            
            modal.find('#planId').val(plan.id);
            modal.find('input[name="name"]').val(plan.name);
            modal.find('input[name="days"]').val(plan.days);
            modal.find('input[name="price"]').val(plan.price);
            modal.find('input[name="description"]').val(plan.description);
            modal.find('input[name="sort_order"]').val(plan.sort_order);
            modal.find('select[name="status"]').val(plan.status);
            
            modal.modal('show');
        });
    });

    // 添加套餐时清空表单
    $('#planModal').on('show.bs.modal', function(e) {
        if (!e.relatedTarget || !e.relatedTarget.classList.contains('edit-plan')) {
            this.querySelector('form').reset();
            this.querySelector('#planId').value = '';
        }
    });
});
</script>

<?php require_once 'layout/footer.php'; ?> 