<?php
/**
 * 朋友圈访客查询系统
 * 
 * @author    锅岛主
 * @link      https://ggdao.net
 * @wechat    IJG55555
 * @copyright Copyright (c) 2026 锅岛主 (https://ggdao.net)
 * @license   保留所有权利
 */

require_once __DIR__ . '/Database.php';

class Admin {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    public function login($username, $password) {
        $sql = "SELECT * FROM admin_users WHERE username = ? AND status = 'active'";
        $admin = $this->db->query($sql, [$username])->fetch();
        
        if ($admin) {
            // 调试信息
            if (DEBUG_MODE) {
                error_log("Login attempt for username: " . $username);
                error_log("Stored hash: " . $admin['password']);
                error_log("Input password hash: " . password_hash($password, PASSWORD_DEFAULT));
            }
            
            if (password_verify($password, $admin['password'])) {
                // 设置管理员会话
                $_SESSION['admin'] = [
                    'id' => $admin['id'],
                    'username' => $admin['username'],
                    'name' => $admin['name']
                ];
                
                // 更新最后登录时间
                $this->db->query(
                    "UPDATE admin_users SET last_login = NOW() WHERE id = ?",
                    [$admin['id']]
                );
                
                return true;
            }
        }
        
        return false;
    }

    public function isAdmin() {
        // 检查是否是通过账号密码登录的管理员
        if (isset($_SESSION['admin'])) {
            return true;
        }
        
        // 检查是否是通过微信登录的管理员
        if (isset($_SESSION['user']) && isset($_SESSION['user']['role']) && $_SESSION['user']['role'] === 'admin') {
            return true;
        }
        
        return false;
    }

    public function checkAccess() {
        if (!$this->isAdmin()) {
            // 如果已经通过微信登录但不是管理员，显示权限不足页面
            if (isset($_SESSION['user'])) {
                header('Location: /admin/unauthorized.php');
                exit;
            }
            // 否则跳转到登录页面
            header('Location: /admin/login.php');
            exit;
        }
    }

    public function logout() {
        unset($_SESSION['admin']);
        session_destroy();
    }

    public function getOverallStats() {
        // 获取总体统计数据
        $stats = [
            'total_users' => 0,
            'total_posts' => 0,
            'total_views' => 0,
            'total_visitors' => 0,
            'today_visitors' => 0,
            'today_visits' => 0,
            'today_posts' => 0
        ];
        
        // 总用户数
        $result = $this->db->query("SELECT COUNT(*) as count FROM users")->fetch();
        $stats['total_users'] = $result['count'];
        
        // 总动态数
        $result = $this->db->query("SELECT COUNT(*) as count FROM posts WHERE status = 'active'")->fetch();
        $stats['total_posts'] = $result['count'];
        
        // 总浏览量和访客数
        $result = $this->db->query("
            SELECT 
                COUNT(*) as total_views,
                COUNT(DISTINCT visitor_id) as total_visitors
            FROM visitors
        ")->fetch();
        $stats['total_views'] = $result['total_views'];
        $stats['total_visitors'] = $result['total_visitors'];
        
        // 今日数据
        $today = date('Y-m-d');
        $result = $this->db->query("
            SELECT 
                COUNT(*) as visit_count,
                COUNT(DISTINCT visitor_id) as visitor_count
            FROM visitors 
            WHERE DATE(visited_at) = ?
        ", [$today])->fetch();
        $stats['today_visits'] = $result['visit_count'];
        $stats['today_visitors'] = $result['visitor_count'];
        
        // 今日被访问的动态数
        $result = $this->db->query("
            SELECT COUNT(DISTINCT post_id) as count 
            FROM visitors 
            WHERE DATE(visited_at) = ?
        ", [$today])->fetch();
        $stats['today_posts'] = $result['count'];
        
        return $stats;
    }

    public function getDailyStats($days = 30) {
        // 获取每日访问统计
        $sql = "
            SELECT 
                DATE(visited_at) as date,
                COUNT(*) as visit_count,
                COUNT(DISTINCT visitor_id) as visitor_count
            FROM visitors
            WHERE visited_at >= DATE_SUB(CURDATE(), INTERVAL ? DAY)
            GROUP BY DATE(visited_at)
            ORDER BY date ASC
        ";
        return $this->db->query($sql, [$days])->fetchAll();
    }

    public function getHourlyStats() {
        // 获取时段访问统计
        $sql = "
            SELECT 
                HOUR(visited_at) as hour,
                COUNT(*) as visit_count
            FROM visitors
            WHERE visited_at >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
            GROUP BY HOUR(visited_at)
            ORDER BY hour ASC
        ";
        return $this->db->query($sql)->fetchAll();
    }

    public function getHotPosts($limit = 5) {
        // 获取热门动态
        $sql = "
            SELECT 
                p.*,
                u.nickname,
                u.avatar,
                COUNT(v.id) as view_count,
                COUNT(DISTINCT v.visitor_id) as visitor_count
            FROM posts p
            JOIN users u ON p.user_id = u.id
            LEFT JOIN visitors v ON p.id = v.post_id
            WHERE p.status = 'active'
            GROUP BY p.id
            ORDER BY view_count DESC
            LIMIT ?
        ";
        return $this->db->query($sql, [$limit])->fetchAll();
    }

    public function getActiveUsers($limit = 5) {
        // 获取活跃用户
        $sql = "
            SELECT 
                u.*,
                COUNT(DISTINCT p.id) as post_count,
                COUNT(v.id) as total_views,
                COUNT(DISTINCT v.visitor_id) as total_visitors
            FROM users u
            LEFT JOIN posts p ON u.id = p.user_id AND p.status = 'active'
            LEFT JOIN visitors v ON p.id = v.post_id
            GROUP BY u.id
            ORDER BY total_views DESC
            LIMIT ?
        ";
        return $this->db->query($sql, [$limit])->fetchAll();
    }

    public function getAllUsers($page = 1, $limit = 10) {
        $offset = ($page - 1) * $limit;
        
        $sql = "
            SELECT 
                u.*,
                (SELECT COUNT(*) FROM users WHERE share_from_user_id = u.id) as referral_count,
                (SELECT COUNT(DISTINCT post_id) FROM visitors v 
                 JOIN posts p ON v.post_id = p.id 
                 WHERE p.user_id = u.id AND v.visitor_id != u.id) as effective_shares,
                (SELECT COUNT(*) FROM posts WHERE user_id = u.id AND status = 'active') as total_posts
            FROM users u
            ORDER BY u.created_at DESC 
            LIMIT ? OFFSET ?
        ";
        
        $stmt = $this->db->query($sql, [$limit, $offset]);
        return $stmt->fetchAll();
    }

    public function getUserCount() {
        $result = $this->db->query("SELECT COUNT(*) as count FROM users")->fetch();
        return $result['count'];
    }

    public function getAllPosts($page = 1, $limit = 20) {
        $offset = ($page - 1) * $limit;
        $sql = "
            SELECT 
                p.*,
                u.nickname,
                u.avatar,
                COUNT(v.id) as view_count,
                COUNT(DISTINCT v.visitor_id) as visitor_count
            FROM posts p
            JOIN users u ON p.user_id = u.id
            LEFT JOIN visitors v ON p.id = v.post_id
            WHERE p.status = 'active'
            GROUP BY p.id
            ORDER BY p.created_at DESC 
            LIMIT ? OFFSET ?
        ";
        return $this->db->query($sql, [$limit, $offset])->fetchAll();
    }

    public function getPostCount() {
        $result = $this->db->query("
            SELECT COUNT(*) as count 
            FROM posts 
            WHERE status = 'active'
        ")->fetch();
        return $result['count'];
    }

    public function deleteUser($userId) {
        // 删除用户的所有动态
        $this->db->query("UPDATE posts SET status = 'deleted' WHERE user_id = ?", [$userId]);
        
        // 删除用户
        return $this->db->query("DELETE FROM users WHERE id = ?", [$userId]);
    }

    public function deletePost($postId) {
        return $this->db->update('posts', 
            ['status' => 'deleted'], 
            'id = ?', 
            [$postId]
        );
    }

    public function getUserPostCount($userId) {
        $sql = "SELECT COUNT(*) as count FROM posts WHERE user_id = ? AND status = 'active'";
        $result = $this->db->query($sql, [$userId])->fetch();
        return $result['count'];
    }

    public function changePassword($currentPassword, $newPassword) {
        // 获取当前管理员信息
        if (isset($_SESSION['admin'])) {
            // 账号密码登录的管理员
            $adminId = $_SESSION['admin']['id'];
            $sql = "SELECT * FROM admin_users WHERE id = ? AND status = 'active'";
            $admin = $this->db->query($sql, [$adminId])->fetch();
            
            if (!$admin || !password_verify($currentPassword, $admin['password'])) {
                return false;
            }

            // 更新新密码
            $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
            $sql = "UPDATE admin_users SET password = ? WHERE id = ?";
            return $this->db->query($sql, [$hashedPassword, $adminId]);
        } else if (isset($_SESSION['user']) && isset($_SESSION['user']['role']) && $_SESSION['user']['role'] === 'admin') {
            // 微信登录的管理员
            $userId = $_SESSION['user']['id'];
            $sql = "SELECT * FROM users WHERE id = ? AND role = 'admin'";
            $user = $this->db->query($sql, [$userId])->fetch();
            
            if (!$user || !password_verify($currentPassword, $user['password'])) {
                return false;
            }

            // 更新新密码
            $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
            $sql = "UPDATE users SET password = ? WHERE id = ?";
            return $this->db->query($sql, [$hashedPassword, $userId]);
        }
        
        return false;
    }
} 