<?php
/**
 * 朋友圈访客查询系统
 * 
 * @author    锅岛主
 * @link      https://ggdao.net
 * @wechat    IJG55555
 * @copyright Copyright (c) 2026 锅岛主 (https://ggdao.net)
 * @license   保留所有权利
 */

require_once __DIR__ . '/../config/config.php';

class Database {
    private static $instance = null;
    private $pdo;
    
    private function __construct() {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4";
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4"
            ];
            
            $this->pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
            
            // 设置会话SQL模式，移除STRICT_TRANS_TABLES
            $this->pdo->exec("SET SESSION sql_mode = 'NO_ENGINE_SUBSTITUTION'");
            
        } catch (PDOException $e) {
            die("数据库连接失败: " . $e->getMessage());
        }
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function getConnection() {
        return $this->pdo;
    }
    
    public function query($sql, $params = []) {
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt;
        } catch (PDOException $e) {
            // 记录错误日志
            error_log("数据库查询错误: " . $e->getMessage());
            error_log("SQL: " . $sql);
            error_log("参数: " . print_r($params, true));
            throw $e;
        }
    }
    
    public function insert($table, $data) {
        $fields = array_keys($data);
        $values = array_fill(0, count($fields), '?');
        
        $sql = "INSERT INTO " . $table . " (" . implode(', ', $fields) . ") 
                VALUES (" . implode(', ', $values) . ")";
        
        return $this->query($sql, array_values($data));
    }
    
    public function update($table, $data, $where, $whereParams = []) {
        $fields = array_map(function($field) {
            return $field . ' = ?';
        }, array_keys($data));
        
        $sql = "UPDATE " . $table . " SET " . implode(', ', $fields);
        
        if ($where) {
            $sql .= " WHERE " . $where;
        }
        
        $params = array_merge(array_values($data), $whereParams);
        return $this->query($sql, $params);
    }
    
    public function delete($table, $where, $params = []) {
        $sql = "DELETE FROM " . $table;
        
        if ($where) {
            $sql .= " WHERE " . $where;
        }
        
        return $this->query($sql, $params);
    }
    
    public function beginTransaction() {
        return $this->pdo->beginTransaction();
    }
    
    public function commit() {
        return $this->pdo->commit();
    }
    
    public function rollBack() {
        return $this->pdo->rollBack();
    }
    
    // 防止对象被克隆
    public function __clone() {
        throw new Exception("Cannot clone singleton");
    }
    
    // 防止对象被反序列化
    public function __wakeup() {
        throw new Exception("Cannot unserialize singleton");
    }
} 