<?php
/**
 * 朋友圈访客查询系统
 * 
 * @author    锅岛主
 * @link      https://ggdao.net
 * @wechat    IJG55555
 * @copyright Copyright (c) 2026 锅岛主 (https://ggdao.net)
 * @license   保留所有权利
 */

require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Post.php';
require_once '../includes/WechatShare.php';

// 获取动态ID
$postId = isset($_GET['id']) ? (int)$_GET['id'] : 0;

// 获取动态详情
$post = new Post();
$postDetails = $post->getPost($postId);

// 如果动态不存在，重定向到首页
if (!$postDetails) {
    header('Location: /');
    exit;
}

// 如果用户未登录，保存当前URL并重定向到登录页面
if (!isset($_SESSION['user'])) {
    $_SESSION['redirect_url'] = $_SERVER['REQUEST_URI'];
    // 记录分享来源
    $_SESSION['share_from'] = [
        'post_id' => $postId,
        'user_id' => $postDetails['user_id']
    ];
    header('Location: /auth/wechat_login.php');
    exit;
}

// 记录访问
$post->recordVisit($postId, $_SESSION['user']['id']);

// 判断当前用户是否是动态发布者
$isOwner = ($postDetails['user_id'] == $_SESSION['user']['id']);

// 只有动态发布者才能看到访客列表
if ($isOwner) {
    $visitors = $post->getVisitors($postId);
}

// 配置微信分享
$wechatShare = new WechatShare();
$currentUrl = "https://{$_SERVER['HTTP_HOST']}{$_SERVER['REQUEST_URI']}";
$shareConfig = $wechatShare->getJsConfig($currentUrl);

// 设置页面标题
$pageTitle = mb_substr($postDetails['content'], 0, 20) . '...';
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $pageTitle; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/@fortawesome/fontawesome-free@5.15.4/css/all.min.css" rel="stylesheet">
    <link href="/assets/css/style.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/viewerjs/1.11.6/viewer.min.css">
    
    <style>
    .post-images {
        display: grid;
        gap: 8px;
        margin: 15px 0;
    }

    .post-images.count-1 {
        grid-template-columns: 1fr;
    }

    .post-images.count-2,
    .post-images.count-4 {
        grid-template-columns: repeat(2, 1fr);
    }

    .post-images.count-3,
    .post-images.count-5,
    .post-images.count-6,
    .post-images.count-7,
    .post-images.count-8,
    .post-images.count-9 {
        grid-template-columns: repeat(3, 1fr);
    }

    .post-image-wrapper {
        position: relative;
        padding-bottom: 100%;
        background: #f5f5f5;
        border-radius: 8px;
        overflow: hidden;
        cursor: pointer;
    }

    .post-images.count-1 .post-image-wrapper {
        padding-bottom: 56.25%;
    }

    .post-image-wrapper img {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        object-fit: cover;
    }
    </style>
</head>
<body>
    <?php ?>
    
    <div class="container mt-4">
        <?php if ($isOwner && $postDetails['share_status'] == 'draft'): ?>
        <!-- 未分享提醒 -->
        <div class="alert alert-warning alert-dismissible fade show" role="alert">
            <h5><i class="fas fa-exclamation-triangle"></i> 动态尚未分享到朋友圈！</h5>
            <div class="alert alert-info mt-3">
                <h6><i class="fas fa-info-circle"></i> 分享步骤：</h6>
                <ol class="mb-0">
                    <li>点击右上角的"<i class="fas fa-ellipsis-v"></i>"按钮</li>
                    <li>选择"分享到朋友圈"</li>
                    <li>在朋友圈编辑页面点击"发表"</li>
                </ol>
            </div>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        <?php endif; ?>
        
        <?php if ($isOwner && $postDetails['share_status'] == 'shared'): ?>
        <!-- 分享成功提示 -->
        <div class="alert alert-success mb-4">
            <h5><i class="fas fa-check-circle"></i> 动态已成功分享！</h5>
            <p class="mb-0">当朋友浏览这条动态时，您可以在下方实时查看访客记录。</p>
        </div>
        <?php endif; ?>
        
        <div class="card">
            <div class="card-body">
                <div class="d-flex align-items-center mb-3">
                    <img src="<?php echo htmlspecialchars($postDetails['avatar']); ?>" 
                         class="rounded-circle me-2" 
                         width="40" height="40" 
                         alt="<?php echo htmlspecialchars($postDetails['nickname']); ?>">
                    <div>
                        <div class="fw-bold"><?php echo htmlspecialchars($postDetails['nickname']); ?></div>
                        <div class="text-muted small">
                            <?php echo date('Y-m-d H:i', strtotime($postDetails['created_at'])); ?>
                        </div>
                    </div>
                </div>
                
                <p class="card-text"><?php echo nl2br(htmlspecialchars($postDetails['content'])); ?></p>
                
                <?php if (!empty($postDetails['image'])): ?>
                <?php 
                $images = explode('|', $postDetails['image']);
                $imageCount = count($images);
                ?>
                <div class="post-images count-<?php echo $imageCount; ?>" id="postImages">
                    <?php foreach ($images as $index => $image): ?>
                    <?php
                    // 确保图片路径是完整的URL
                    $imageUrl = strpos($image, 'http') === 0 ? $image : "https://{$_SERVER['HTTP_HOST']}" . $image;
                    ?>
                    <div class="post-image-wrapper" onclick="previewImage(<?php echo $index; ?>)">
                        <img src="<?php echo htmlspecialchars($imageUrl); ?>" alt="动态图片" data-index="<?php echo $index; ?>">
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
                
                <div class="d-flex justify-content-between align-items-center text-muted small">
                    <?php if ($isOwner): ?>
                    <div>
                        <i class="fas fa-eye me-1"></i> <?php echo number_format($postDetails['view_count']); ?> 次浏览
                        <span class="mx-2">·</span>
                        <i class="fas fa-users me-1"></i> <?php echo count($visitors); ?> 位访客
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- 访客列表 - 只对发布者显示 -->
        <?php if ($isOwner && $visitors): ?>
        <div class="card mt-4">
            <div class="card-header">
                <h5 class="card-title mb-0">访客列表</h5>
            </div>
            <div class="card-body">
                <div class="row g-3">
                    <?php foreach ($visitors as $visitor): ?>
                    <div class="col-6 col-md-4 col-lg-3">
                        <div class="d-flex align-items-center">
                            <img src="<?php echo htmlspecialchars($visitor['avatar']); ?>" 
                                 class="rounded-circle me-2" 
                                 width="32" height="32" 
                                 alt="<?php echo htmlspecialchars($visitor['nickname']); ?>">
                            <div class="small">
                                <div><?php echo htmlspecialchars($visitor['nickname']); ?></div>
                                <div class="text-muted">
                                    <?php echo date('m-d H:i', strtotime($visitor['visited_at'])); ?>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        <?php endif; ?>

      
    </div>

    <!-- 微信JS SDK -->
    <script src="https://res.wx.qq.com/open/js/jweixin-1.6.0.js"></script>
    <script>
    // 准备分享图片
    var shareImgUrl = '<?php 
        if (!empty($postDetails["image"])) {
            // 如果动态有图片，使用第一张图片
            $images = explode('|', $postDetails["image"]);
            echo "https://{$_SERVER['HTTP_HOST']}" . $images[0];
        } else {
            // 否则使用默认分享图片
            echo "https://{$_SERVER['HTTP_HOST']}/assets/images/share-logo.png";
        }
    ?>';

    wx.config({
        debug: false,
        appId: '<?php echo $shareConfig["appId"]; ?>',
        timestamp: <?php echo $shareConfig["timestamp"]; ?>,
        nonceStr: '<?php echo $shareConfig["nonceStr"]; ?>',
        signature: '<?php echo $shareConfig["signature"]; ?>',
        jsApiList: [
            'updateAppMessageShareData',
            'updateTimelineShareData',
            'onMenuShareTimeline',
            'onMenuShareAppMessage',
            'previewImage'
        ]
    });

    // 添加图片预览函数
    function previewImage(index) {
        const images = Array.from(document.querySelectorAll('#postImages img')).map(img => img.src);
        wx.previewImage({
            current: images[index],
            urls: images
        });
    }

    // 添加分享到朋友圈的函数
    function shareToTimeline() {
        // 先更新分享状态
        updateShareStatus().then(() => {
            // 状态更新成功后，触发微信分享
            wx.updateTimelineShareData({
                title: '<?php echo htmlspecialchars(mb_substr($postDetails['content'], 0, 50)); ?>',
                link: '<?php echo $currentUrl; ?>',
                imgUrl: shareImgUrl,
                success: function() {
                    // 分享成功后刷新页面
                    setTimeout(() => {
                        window.location.reload();
                    }, 1000);
                }
            });
        });
    }

    wx.ready(function() {
        // 配置分享给朋友
        wx.updateAppMessageShareData({ 
            title: '<?php echo htmlspecialchars(mb_substr($postDetails['content'], 0, 30)); ?>',
            desc: '这是我给你发的条朋友圈，抽空看看来我的朋友圈吧~~',
            link: '<?php echo $currentUrl; ?>',
            imgUrl: shareImgUrl,
            success: function() {
                if (<?php echo $isOwner ? 'true' : 'false'; ?>) {
                    updateShareStatus();
                }
            }
        });
        
        // 配置分享到朋友圈
        wx.updateTimelineShareData({ 
            title: '<?php echo htmlspecialchars(mb_substr($postDetails['content'], 0, 50)); ?>',
            link: '<?php echo $currentUrl; ?>',
            imgUrl: shareImgUrl,
            success: function() {
                if (<?php echo $isOwner ? 'true' : 'false'; ?>) {
                    updateShareStatus();
                }
            }
        });

        // 兼容旧版本的分享接口
        wx.onMenuShareTimeline({
            title: '<?php echo htmlspecialchars(mb_substr($postDetails['content'], 0, 50)); ?>',
            link: '<?php echo $currentUrl; ?>',
            imgUrl: shareImgUrl,
            success: function() {
                if (<?php echo $isOwner ? 'true' : 'false'; ?>) {
                    updateShareStatus();
                }
            }
        });

        wx.onMenuShareAppMessage({
            title: '<?php echo htmlspecialchars(mb_substr($postDetails['content'], 0, 30)); ?>',
            desc: '这是我给你发的条朋友圈，抽空看看来我的朋友圈吧~~',
            link: '<?php echo $currentUrl; ?>',
            imgUrl: shareImgUrl,
            type: '',
            dataUrl: '',
            success: function() {
                if (<?php echo $isOwner ? 'true' : 'false'; ?>) {
                    updateShareStatus();
                }
            }
        });
    });

    // 更新分享状态的函数
    function updateShareStatus() {
        return fetch('/posts/update_share_status.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                post_id: <?php echo $postId; ?>,
                share_status: 'shared'
            })
        })
        .then(response => response.json())
        .then(data => {
            if (!data.success) {
                throw new Error(data.message || '更新失败');
            }
            return data;
        })
        .then(() => {
            // 更新成功后刷新页面
            window.location.reload();
        })
        .catch(error => {
            console.error('Error:', error);
        });
    }
    </script>

    <?php  ?> 