<?php

namespace App\Http\Controllers\Admin\Admin;

use App\Models\Admin\Role;
use Illuminate\Http\Request;
use App\Models\Admin\Permission;
use Illuminate\Support\Facades\View;
use App\Http\Controllers\BaseController;
use Illuminate\Support\Facades\Response;
use App\Http\Requests\Admin\Admin\RoleCreateRequest;

class RoleController extends BaseController
{
    /** 实例 */
    protected $model;

    /** 初始化 */
    public function __construct()
    {
        parent::__construct();
        // 初始化实例
        $this->model = new Role();
    }

    /**
     * 角色列表
     * @return \Illuminate\Contracts\View\View
     */
    public function index()
    {
        return View::make('admin.admin.role.index');
    }

    /**
     * 角色列表接口数据
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function data(Request $request)
    {
        $res = $this->model->paginate($request->get('limit', 30));
        $data = [
            'code' => 0,
            'msg' => '正在请求中...',
            'count' => $res->total(),
            'data' => $res->items()
        ];
        return Response::json($data);
    }

    /**
     * 添加角色
     * @return \Illuminate\Contracts\View\View
     */
    public function create()
    {
        $model = $this->model;
        return View::make('admin.admin.role.create', compact('model'));
    }

    /**
     * 添加角色
     * @param RoleCreateRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(RoleCreateRequest $validate)
    {
        $data = $this->request->only(['name', 'display_name']);
        if (!$validate->check($data)) {
            return $this->error($validate->getError());
        }
        try {
            $this->model->create($data);
            return $this->success('添加成功');
        } catch (\Exception $e) {
            return $this->error('添加失败');
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * 更新角色
     * @return \Illuminate\Contracts\View\View
     */
    public function edit()
    {
        $model = $this->model->findOrFail($this->request->id);
        return View::make('admin.admin.role.edit', compact('model'));
    }

    /**
     * 更新角色
     * @param RoleCreateRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(RoleCreateRequest $validate)
    {
        $role = $this->model->findOrFail($this->request->id);
        $data = $this->request->only(['name', 'display_name']);
        if (!$validate->check($data)) {
            return $this->error($validate->getError());
        }
        try {
            $role->update($data);
            return $this->success('更新成功');
        } catch (\Exception $e) {
            return $this->error('更新失败');
        }
    }

    /**
     * 删除角色
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy(Request $request)
    {
        $ids = $request->get('ids');
        if (!is_array($ids) || empty($ids)) {
            return $this->error('请选择删除项');
        }
        // Admin 角色不可删除
        if (in_array(1, $ids)) {
            return $this->error('包含角色ID1不可删除,请重新选择');
        }
        try {
            $this->model->destroy($ids);
            return $this->success('删除成功');
        } catch (\Exception $e) {
            return $this->error('删除失败');
        }
    }

    /**
     * 分配权限
     * @param Request $request
     * @return \Illuminate\Contracts\View\View
     */
    public function permission()
    {
        $role = $this->model->findOrFail($this->request->id);
        $permissions = Permission::with('allChilds')->orderByRaw('sort asc')->where('parent_id', 0)->get();
        foreach ($permissions as $p1) {
            $p1->own = $role->hasPermissionTo($p1->id) ? 'checked' : false;
            if ($p1->childs->isNotEmpty()) {
                foreach ($p1->childs as $p2) {
                    $p2->own = $role->hasPermissionTo($p2->id) ? 'checked' : false;
                    if ($p2->childs->isNotEmpty()) {
                        foreach ($p2->childs as $p3) {
                            $p3->own = $role->hasPermissionTo($p3->id) ? 'checked' : false;
                        }
                    }
                }
            }
        }
        return View::make('admin.admin.role.permission', compact('role', 'permissions'));
    }

    /**
     * 存储权限
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function assignPermission()
    {
        $role = $this->model->findOrFail($this->request->id);
        // Admin 角色不可编辑权限
        // if ($id == 1) {
        //     return Redirect::to(URL::route('admin.role'))->withErrors(['success' => '该角色不可编辑权限']);
        // }
        $permissions = $this->request->get('permissions', []);
        try {
            $role->syncPermissions($permissions);
            return $this->success('更新成功');
        } catch (\Exception $e) {
            return $this->error('更新失败');
        }
    }
}
