<?php

namespace App\Http\Controllers\Api\V1\Shop;

use App\ToolsClass\RedisTool;
use App\Common\Enum\RedisEnum;
use App\Services\Shop\GoodsModel;
use App\Services\Shop\CollectModel;
use App\Http\Controllers\BaseController;
use App\Http\Resources\Shop\GoodsInfoResources;
use App\Http\Resources\Shop\GoodsListResources;

class GoodsController extends BaseController
{
    // 商品列表
    public function list(GoodsModel $model)
    {
        // 筛选条件
        $where = [];
        if (isset($this->request->cat_id)) {
            $where[] = ['cat_id', '=', $this->request->cat_id];
        }
        if (isset($this->request->brand_id)) {
            $where[] = ['brand_id', '=', $this->request->brand_id];
        }
        if (isset($this->request->keywords)) {
            $where[] = ['title', 'like', '%' . $this->request->keywords . '%'];
        }
        // 排序
        $sort_type = $this->request->sort_type ?? 1;
        if ($sort_type == 1) {
            $sort = 'sort desc,id desc';    // 综合排序
        } else if ($sort_type == 2) {
            $sort = 'id desc,sort desc';    // 最新上架
        } else if ($sort_type == 3) {
            $sort = 'sales desc,id desc';    // 销量优先
        } else if ($sort_type == 4) {
            $sort = 'price asc,id desc';    // 价格排序（低到高）
        } else if ($sort_type == 5) {
            $sort = 'price desc,id desc';    // 价格排序（高到低）
        }
        $list = $model->factor()
            ->where($where)
            ->orderByRaw($sort)
            ->paginate($this->request->per_page ?? 20);
        $list = $model->page($list);
        $list['data'] = GoodsListResources::collection($list['data']);
        return $this->success('请求成功', $list);
    }

    /**
     * 商品详情
     *
     * @param  \App\Services\Shop\GoodsModel $goodsModel
     */
    public function info(GoodsModel $goodsModel)
    {
        $data = $goodsModel->factor()->where('id', $this->request->id)->first();
        if (!$data) {
            return $this->error('商品不存在或已下架!');
        }
        $data = new GoodsInfoResources($data);
        return $this->success('请求成功', $data);
    }

    /**
     * 收藏，取消商品
     *
     * @param  \App\Services\Shop\GoodsModel   $goodsModel
     * @param  \App\Services\Shop\CollectModel $collectModel
     * @param  \App\ToolsClass\RedisTool       $redis
     */
    public function collect(GoodsModel $goodsModel, CollectModel $collectModel, RedisTool $redis)
    {
        $user = auth('api')->user();
        $key = RedisEnum::SHOP_COLLECT . $user->id;
        if ($redis->block($key, 1)) {
            $data = $goodsModel->factor()->where('id', $this->request->id)->first();
            if (!$data) {
                return $this->unlock($redis, $key)->error('商品不存在或已下架!');
            }
            $res = $collectModel->collect($user, $data->id);
            if (!$res) {
                return $this->error('操作失败!');
            }
            return $this->success('操作成功!');
        }
        return $this->error('操作过快!');
    }
}
