<?php

namespace App\Services\Order;

use App\Models\Order\Order;
use App\Services\Box\BoxModel;
use App\Common\Method\Response;
use App\Services\Shop\GoodsModel;
use App\Services\User\UsersModel;
use App\Services\User\GoldLogModel;
use App\Services\User\MoneyLogModel;
use App\ToolsClass\WxPay\MiniPayTool;

class OrderModel extends Order
{
    /** 支付失败 */
    const STA_MIN_1 = -1;
    /** 待支付 */
    const STA_0 = 0;
    /** 已完成 */
    const STA_1 = 1;
    /** 支付状态 */
    static public $status = [
        self::STA_MIN_1 => '支付失败',
        self::STA_0 => '待支付',
        self::STA_1 => '已完成',
    ];

    /** 未退款 */
    const REF_0 = 0;
    /** 待退款 */
    const REF_1 = 1;
    /** 已退款 */
    const REF_2 = 2;
    /** 退款状态 */
    static public $refund_status = [
        self::REF_0 => '未退款',
        self::REF_1 => '待退款',
        self::REF_2 => '已退款',
    ];

    /** 抽盒 */
    const TYPE_1 = 1;
    /** 发货 */
    const TYPE_2 = 2;
    /** 商城 */
    const TYPE_3 = 3;
    /** 订单类型 */
    static public $type = [
        self::TYPE_1 => '抽盒',
        self::TYPE_2 => '发货',
        self::TYPE_3 => '商城',
    ];

    /** 购物金支付 */
    const P_TYPE_1 = 1;
    /** 余额支付 */
    const P_TYPE_2 = 2;
    /** 微信支付 */
    const P_TYPE_3 = 3;
    /** 支付宝支付 */
    const P_TYPE_4 = 4;
    /** 支付类型 */
    static public $pay_type = [
        self::P_TYPE_1 => '购物金支付',
        self::P_TYPE_2 => '余额支付',
        self::P_TYPE_3 => '微信支付',
        self::P_TYPE_4 => '支付宝支付',
    ];
    /*
     * 查询订单状态支付宝当面付使用
     */
    public function getOrderStatus($order_sn)
    {
        $model = new self();
        $order = $model->where('order_sn',$order_sn)->first();
        if(empty($order)){
            return false;
        }
        if($order->status == self::STA_0){
            return false;
        }
        return true;
    }

    /**
     * 生成订单号
     */
    public function order_sn()
    {
        $order_sn = orderNum();
        $coupon = $this->where(['order_sn' => $order_sn])->first();
        if ($coupon) {
            return $this->order_sn;
        }
        return $order_sn;
    }

    /**
     * 抵扣计算
     *
     * @param  object $user         用户数据
     * @param  string $total_money  总金额
     * @param  object $param         数据[
     *          pay_type        支付方式       必填
     *          coin            是否购物金抵扣 必填
     *          money           是否余额抵扣   必填
     * ]
     */
    public function discount($user, $total_money, $param)
    {
        $data['total_money'] = $total_money; // 总金额
        $data['pay_money'] = $total_money;  // 支付金额
        $data['gold_dis'] = 0;              // 购物金抵扣金额
        $data['money_dis'] = 0;             // 余额抵扣金额
        $pay_type = $param['pay_type'];
        // 支付金额判断
        if ($total_money >= 0.01) {
            // 微信支付
            if (in_array($pay_type, [self::P_TYPE_3,self::P_TYPE_4])) {
                // 余额抵扣
                if (isset($param['money']) && $param['money'] && $data['pay_money'] >= 0.01) {
                    $moneyLogModel = new MoneyLogModel();
                    $moneyDis = $moneyLogModel->discount($data['pay_money'], $user->userinfo);
                    if (!$moneyDis) {
                        return Response::error(Response::getMessage());
                    }
                    $data['money_dis'] = $moneyDis['money_dis'];             // 余额抵扣金额
                    $data['pay_money'] = $moneyDis['pay_money'];            // 实际支付金额
                }
                // 购物金抵扣
                if (isset($param['gold']) && $param['gold'] && $data['pay_money'] >= 0.01) {
                    $goldModel = new GoldLogModel();
                    $goldDis = $goldModel->discount($data['pay_money'], $user->userinfo);
                    if (!$goldDis) {
                        return Response::error(Response::getMessage());
                    }
                    $data['gold_dis'] = $goldDis['gold_dis'];              // 购物金抵扣金额
                    $data['pay_money'] = $goldDis['pay_money'];            // 实际支付金额
                }
            }
        }
        return Response::success($data);
    }

    /**
     * 支付
     *
     * @param  object $user     用户数据
     * @param  string $type     订单类型
     * @param  array  $param    数据[
     *          pay_type        支付方式       必填
     *          gold            是否购物金抵扣 必填
     *          money          是否余额抵扣    必填
     * @param  array  $discount[
     *          total_money     支付金额
     *          pay_money       支付金额
     *          gold_dis        购物金抵扣金额
     *          money_dis       余额抵扣金额
     * ]
     * @param  object $request  请求
     * @param  array  $data     其他数据
     */
    public function pay($user, $type, $param, $discount, $data = null)
    {
        // 支付方式判断
        $pay_type = $param['pay_type'];
        if (in_array($pay_type, [self::P_TYPE_3,self::P_TYPE_4])) {
            if ($discount['pay_money'] <= 0) {
                if (isset($param['money']) && $param['money']) {
                    $pay_type = self::P_TYPE_2;
                }
                if (isset($param['gold']) && $param['gold']) {
                    $pay_type = self::P_TYPE_1;
                }
            }
            // 余额支付（使用余额支付并且抵扣大于0.01）
            if (isset($param['money']) && $param['money'] && $discount['money_dis'] >= 0.01) {
                if ($user->userinfo->money < $discount['money_dis']) {
                    return Response::error('余额不足!');
                }
                $money = new MoneyLogModel();
                $money_log = $money->log($user->userinfo, -$discount['money_dis'], $money::TYPE_3);
                if (!$money_log) {
                    return Response::error(Response::getMessage());
                }
            }
            // 购物金支付（使用购物金支付并且抵扣大于0.01）
            if (isset($param['gold']) && $param['gold'] && $discount['gold_dis'] >= 0.01) {
                if ($user->userinfo->gold < $discount['gold_dis']) {
                    return Response::error('购物金不足!');
                }
                $gold = new GoldLogModel();
                $gold_log = $gold->log($user->userinfo, -$discount['gold_dis'], $gold::TYPE_3);
                if (!$gold_log) {
                    return Response::error(Response::getMessage());
                }
            }
        }
        $order = new self();
        $order->order_sn = $this->order_sn();
        $order->user_id = $user->id;
        $order->data_id = $data['id'] ?? null;
        $order->type = $type;
        $order->pay_type = $pay_type;
        $order->total_money = $discount['total_money'];
        $order->money_dis = $discount['money_dis'];
        $order->gold_dis = $discount['gold_dis'];
        $order->pay_money = $discount['pay_money'];
        $order->data = $data;
        if (!$order->save()) {
            return Response::error('订单生成失败!');
        }
        // 完全抵扣
        if ($order->pay_money <= 0) {
            $result = $this->notify($order);
            if (!$result) {
                return Response::error('支付失败!');
            }
        }
        return Response::success($order);
    }

    /**
     * 取消支付（微信支付）
     *
     * @param  object $order
     */
    public function cancelPay($order, $status = true)
    {
        // 公共部分
        $usersModel = new UsersModel();
        $user  = $usersModel->find($order->user_id);
        // 订单状态改变
        if ($status) {
            $order->status = self::STA_MIN_1;
            if (!$order->save()) {
                return Response::error('订单状态改变失败!');
            }
        }
        if ($user) {
            // 购物金退回
            $gold = new GoldLogModel();
            if ($order->gold_dis > 0) {
                $gold_log = $gold->log($user->userinfo, $order->gold_dis, $gold::TYPE_4);
                if (!$gold_log) {
                    return Response::error('购物金退回失败!');
                }
            }
            // 余额退回
            $money = new MoneyLogModel();
            if ($order->money_dis > 0) {
                $money_log = $money->log($user->userinfo, $order->money_dis, $money::TYPE_4);
                if (!$money_log) {
                    return Response::error('余额退回失败!');
                }
            }
            // 抽奖订单
            if ($order->type == self::TYPE_1) {
                if (!BoxModel::cancelPay($order)) {
                    return Response::error(Response::getMessage());
                }
            }
            // 发货
            if ($order->type == self::TYPE_2) {
                if (!ShipModel::cancelPay($order)) {
                    return Response::error(Response::getMessage());
                }
            }
            // 商城
            if ($order->type == self::TYPE_3) {
                if (!GoodsModel::cancelPay($order)) {
                    return Response::error(Response::getMessage());
                }
            }
        }
        return Response::success();
    }

    /**
     * 回调
     *
     * @param  object $order
     */
    public function notify($order)
    {
        // 公共部分
        //修改订单状态
        $usersModel = new UsersModel();
        $user = $usersModel->where('id', $order->user_id)->first();
        if ($order->status != self::STA_0) {
            return Response::error('订单状态不正确!');
        }
        $order->status = self::STA_1;
        $order->pay_time = time();
        if (!$order->save()) {
            return Response::error('订单状态修改失败!');
        }
        if ($user) {
            // 抽奖订单
            if ($order->type == self::TYPE_1) {
                if (!BoxModel::notify($order, $user)) {
                    return Response::error(Response::getMessage());
                }
            }
            // 发货
            if ($order->type == self::TYPE_2) {
                if (!ShipModel::notify($order, $user)) {
                    return Response::error(Response::getMessage());
                }
            }
            // 商城
            if ($order->type == self::TYPE_3) {
                if (!GoodsModel::notify($order, $user)) {
                    return Response::error(Response::getMessage());
                }
            }
        }
        return Response::success();
    }

    /** 退款 */
    public function refund($order)
    {
        // 取消支付接口
        $result = $this->cancelPay($order, false);
        // 状态改变
        $order->refund_status = self::REF_2;
        $orderRes = $order->save();
        // 退款
        $res = true;
        if ($order->pay_money > 0) {
            // 微信退款
            if ($order->pay_type == OrderModel::P_TYPE_3) {
                $pay = new MiniPayTool();
                $res = $pay->refund($order);
            }
        }
        if (!$result || !$orderRes || !$res) {
            return Response::error('退款失败!');
        }
        return Response::success('退款成功!');
    }

    /**
     * 过期订单处理(下单了余额抵扣了微信未唤起来，如果微信唤起来直接是退款订单)
     */
    public function expired()
    {
        $list = $this->where('status', self::STA_0)->get();
        if (!$list->isEmpty()) {
            foreach ($list as $v) {
                // 判断时间(5分钟内)
                $second = 5;
                if ((strtotime($v->created_at) + 60 * $second) < time()) {
                    $this->cancelPay($v);
                }
            }
        }
    }
}
