<?php

namespace App\Services\Shop;

use App\Models\Shop\Goods;
use App\Common\Method\Response;
use App\Services\Order\OrderListModel;
use App\Services\Order\ShipModel;
use Illuminate\Support\Facades\DB;

/** 商城 */
class GoodsModel extends Goods
{
    /** 统一规格 */
    const UNIFIED_SPECIFICATION  = 0;
    /** 多规格 */
    const MULTIPLE_SPECIFICATIONS  = 1;
    /** 类型 */
    static public $isAttributeOptions = [
        self::UNIFIED_SPECIFICATION => '统一规格',
        self::MULTIPLE_SPECIFICATIONS => '多规格',
    ];

    /** 公共条件 */
    public function factor()
    {
        return $this->where(['status' => 1]);
    }

    /**
     * 增加/更新商品
     * @param $post 表单提交的数据
     * @param null|Product $productModel 商品模型 有值表示更新
     * @return string|null
     */
    public static function addOrEdit($post, $productModel = null)
    {
        DB::beginTransaction();
        try {
            if (!$productModel) {
                //创建商品基础信息
                $productModel = new self();
            }
            //更新基本信息
            $productModel->is_attribute = $post['is_attribute'];
            $productModel->title = $post['title'];
            $productModel->cat_id = $post['cat_id'];
            $productModel->brand_id = $post['brand_id'];
            $productModel->thumb = $post['thumb'];
            $productModel->status = $post['status'];
            $productModel->sort = $post['sort'];
            $productModel->editor = $post['editor'] ?? '';
            // 单规格直接赋值
            if ($post['is_attribute'] == self::UNIFIED_SPECIFICATION) {
                $productModel->old_price = $post['old_price'];
                $productModel->price = $post['price'];
                $productModel->sales = $post['sales'];
                $productModel->stock = $post['stock'];
            }
            $productModel->save($post);

            //多规格添加
            if ($post['is_attribute'] == self::MULTIPLE_SPECIFICATIONS) {
                // 清除原来的规格相关数据
                SpecModel::where('goods_id', $productModel->id)->delete();
                SpecValModel::where('goods_id', $productModel->id)->delete();
                SkuModel::where('goods_id', $productModel->id)->delete();
                // 判断
                if (empty($post['skus'])) {
                    return Response::error('SKU数据不能为空');
                }
                $specValueGroups = [];
                $specValueIdMap = [];
                foreach ($post['spec'] as $spec) {
                    //商品规格
                    $specModel = SpecModel::create(['goods_id' => $productModel->id, 'title' => $spec['title']]);
                    $specValueIdGroup = [];
                    foreach ($spec['child'] as $specValue) {
                        //商品规格值
                        $specValueModel = SpecValModel::create([
                            'goods_id' => $productModel->id,
                            'spec_id' => $specModel->id,
                            'title' => $specValue['title'],
                            'is_checked' => $specValue['checked'] === 'true' ? 1 : 2,
                        ]);
                        if ($specValue['checked'] === 'true') {
                            $specValueIdMap[$specValue['id']] = [
                                'id' => $specValue['id'],
                                'title' => $specValue['title'],
                                'real_id' => $specValueModel->id
                            ];
                            $specValueIdGroup[] = $specValue['id'];
                        }
                    }
                    if ($specValueIdGroup) {
                        $specValueGroups[] = $specValueIdGroup;
                    }
                }
                $totalSales = 0;    // 总库量
                $totalStock = 0;    // 总库存
                $minPrice = $minMarketPrice  = 9999999999;
                foreach (self::diker($specValueGroups) as $diker) {
                    $skuName = [];
                    $skuData = [];
                    $skuIndexData = [];
                    if (is_array($diker)) {
                        //多规格
                        foreach ($diker as $v) {
                            $skuName[] = $specValueIdMap[$v]['title'];
                            $skuData[] = $specValueIdMap[$v]['real_id'];
                            $skuIndexData[] = $specValueIdMap[$v]['id'];
                        }
                    } else {
                        //单规格
                        $skuName[] = $specValueIdMap[$diker]['title'];
                        $skuData[] = $specValueIdMap[$diker]['real_id'];
                        $skuIndexData[] = $specValueIdMap[$diker]['id'];
                    }

                    $skuIndexStr = join('-', $skuIndexData);
                    if ($post['skus'][$skuIndexStr]['price'] < $minPrice) {
                        $minPrice = $post['skus'][$skuIndexStr]['price'];
                    }
                    if ($post['skus'][$skuIndexStr]['old_price'] < $minMarketPrice) {
                        $minMarketPrice = $post['skus'][$skuIndexStr]['old_price'];
                    }
                    //商品sku
                    SkuModel::create([
                        'goods_id' => $productModel->id,
                        'name' => join(';', $skuName),
                        'thumb' => imageUrl($post['skus'][$skuIndexStr]['thumb'], false),
                        'price' => $post['skus'][$skuIndexStr]['price'],
                        'old_price' => $post['skus'][$skuIndexStr]['old_price'],
                        'sales' => $post['skus'][$skuIndexStr]['sales'],
                        'stock' => $post['skus'][$skuIndexStr]['stock'],
                        'data' => join('-', $skuData),
                        'status' => $post['skus'][$skuIndexStr]['status'],
                    ]);
                    $totalSales += $post['skus'][$skuIndexStr]['sales'];
                    $totalStock += $post['skus'][$skuIndexStr]['stock'];
                }
                //多库存重新更新商品基础信息
                $productModel->sales = $totalSales;
                $productModel->stock = $totalStock;
                $productModel->price = $minPrice;
                $productModel->old_price = $minMarketPrice;
                $productModel->save();
            }
            DB::commit();
            return Response::success();
        } catch (\Exception $e) {
            DB::rollback();
            return Response::error('操作失败!');
        }
    }

    /**
     * 获取商品规格相关数据
     * @param Product $productInfo 商品信息
     * @return array
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     */
    public static function makeSkuTableData($productInfo)
    {
        if ($productInfo['is_attribute']) {
            $specArr = SpecModel::where('goods_id', $productInfo['id'])->get()->toArray();
            $specValueMap = self::setArray2Index(
                SpecValModel::whereIn('spec_id', array_column($specArr, 'id'))->get()->toArray(),
                'spec_id'
            );
            $specData = [];
            foreach ($specArr as $item) {
                $specData[] = [
                    'id' => $item['id'],
                    'title' => $item['title'],
                    'child' => array_map(function ($v) {
                        return [
                            'id' => $v['id'],
                            'title' => $v['title'],
                            'checked' => $v['is_checked'] == 1,
                        ];
                    }, $specValueMap[$item['id']]),
                ];
            }
            $skuArr = SkuModel::where('goods_id', $productInfo['id'])->get()->toArray();
            $skuData = [];
            foreach ($skuArr as $item) {
                $skuData['skus[' . $item['data'] . '][thumb]'] = $item['thumb'];
                $skuData['skus[' . $item['data'] . '][old_price]'] = $item['old_price'];
                $skuData['skus[' . $item['data'] . '][price]'] = $item['price'];
                $skuData['skus[' . $item['data'] . '][sales]'] = $item['sales'];
                $skuData['skus[' . $item['data'] . '][stock]'] = $item['stock'];
                $skuData['skus[' . $item['data'] . '][status]'] = $item['status'];
            }
        } else {
            $specData = [];
            $skuData = [
                'old_price' => $productInfo['old_price'],
                'price' => $productInfo['price'],
                'sales' => $productInfo['sales'],
                'stock' => $productInfo['stock'],
                'status' => $productInfo['status'],
            ];
        }
        $data = ['specData' => $specData, 'skuData' => $skuData];
        return Response::success($data);
    }

    /**
     * @param array $arr
     * @param $key
     * @return array
     */
    public static function setArray2Index(array $arr, $key)
    {
        $newArr = [];
        foreach ($arr as $k => $v) {
            if (is_array($v)) {
                $newArr[$v[$key]][] = $v;
            }
        }

        return $newArr;
    }

    /**
     * 笛卡尔积
     * @param $arr
     * @return array|mixed|null
     */
    static public function diker($arr)
    {
        $result = array_shift($arr);
        while ($curArr = array_shift($arr)) {
            $lastArr = $result;
            $result = array();
            foreach ($lastArr as $lastVal) {
                if (!is_array($lastVal)) {
                    $lastVal = array($lastVal);
                }
                foreach ($curArr as $curVal) {
                    if (!is_array($curVal)) {
                        $curVal = array($curVal);
                    }
                    $result[] =  array_merge_recursive($lastVal, $curVal);
                }
            }
        }
        return $result;
    }

    /**
     * 发货取消支付
     *
     * @param  object $order
     */
    public static function cancelPay($order)
    {
        // 规格&商品库存增加，销量减少
        $skuModel = new SkuModel();
        $goodsModel = new GoodsModel();
        $sku = $skuModel->where('id', $order->data['sku_id'])->first();
        $goods = $goodsModel->where('id', $order->data['id'])->first();
        if ($sku && $goods) {
            $num = $order->data['num'];
            $goods->sales += $num;
            $goods->stock -= $num;
            $sku->sales += $num;
            $sku->stock -= $num;
            if (!$goods->save() || !$sku->save()) {
                return Response::error('库存增加失败!');
            }
        }
        return Response::success();
    }

    /**
     * 发货支付回调
     *
     * @param  object $order
     * @param  object $user
     */
    public static function notify($order, $user)
    {
        // 奖品入库
        $orderListModel = new OrderListModel();
        $orderListRes = $orderListModel->ShopList($order, $orderListModel::G_TYPE_1);
        if (!$orderListRes) {
            return Response::error(Response::getMessage());
        }
        // 发货表数据
        $log = new ShipModel();
        $log->order_list_id = $orderListRes->id;
        $log->express_money = $order->data['express_money'];
        $log->name = $order->data['name'];
        $log->mobile = $order->data['mobile'];
        $log->address = $order->data['address'];
        $log->ramk = $order->data['ramk'];
        if (!$log->save()) {
            return Response::error('发货信息录入失败!');
        }
        return Response::success();
    }
}
