<?php

namespace App\Services\Sms;

use Exception;
use App\Models\Sms\SmsLog;
use App\Models\User\Users;
use Overtrue\EasySms\EasySms;
use App\Common\Method\Response;
use App\Exceptions\ApiException;
use App\Models\Admin\Configuration;
use Illuminate\Support\Facades\DB;

/** 短信发送 */
class SmsLogModel extends SmsLog
{
    // 短信配置
    protected $config;
    protected $sms_config;

    /** 注册 */
    const TYPE_1 = 1;
    /** 登录 */
    const TYPE_2 = 2;
    /** 忘记密码 */
    const TYPE_3 = 3;
    /** 状态描述 */
    public static $type = [
        self::TYPE_1 => '注册',
        self::TYPE_2 => '登录',
        self::TYPE_3 => '忘记密码',
    ];

    /** 未使用 */
    const STA_0 = 1;
    /** 已使用 */
    const STA_1 = 2;
    /** 状态描述 */
    public static $status = [
        self::STA_0 => '未使用',
        self::STA_1 => '已使用',
    ];

    /** 初始化配置 */
    public function __construct()
    {
        parent::__construct();
        $config = new Configuration();
        $this->config = $config;
        // 初始化配置
        $this->sms_config = [
            // HTTP 请求的超时时间（秒）
            'timeout' => 5.0,
            // 默认发送配置
            'default' => [
                // 网关调用策略，默认：顺序调用
                'strategy' => \Overtrue\EasySms\Strategies\OrderStrategy::class,
                // 默认可用的发送网关
                'gateways' => [$config->getVal('sms_type')],
            ],
            // 可用的网关配置
            'gateways' => [
                'errorlog' => [
                    'file' => '/tmp/easy-sms.log',
                ],
                'aliyun' => [
                    'access_key_id'     =>  $config->getVal('sms_aliyun_access_key'),
                    'access_key_secret' =>  $config->getVal('sms_aliyun_secret_key'),
                    'sign_name'         =>  $config->getVal('sms_aliyun_signname'),
                ],
                'qcloud' => [
                    'sdk_app_id' =>  $config->getVal('sms_qcloud_access_key'),
                    'app_key'    =>  $config->getVal('sms_qcloud_secret_key'),
                    'sign_name'  =>  $config->getVal('sms_qcloud_signname'),
                ],
            ],
        ];
    }

    /**
     * 发送短信
     *
     * @param  string $mobile
     * @param  string $type
     */
    public function send($mobile, $type)
    {
        DB::beginTransaction();
        try {
            //验证码
            $code = mt_rand(1000, 9999);
            //判断发送频率(一分钟)
            $minute = 1;
            $sms = $this->where(['mobile' => $mobile, 'status' => self::STA_0, 'type' => $type])
                ->orderBy('created_at', 'desc')
                ->first();
            if ($sms && ((time() - $sms->created_at->format('U')) < 60 * $minute)) {
                return Response::error('验证码发送过快!');
            }
            //判断事件类型
            $user = Users::where('mobile', $mobile)->first();
            if ($type == $this::TYPE_1 && $user) {
                return Response::error('该手机号已被注册!');
            }
            if ($type != $this::TYPE_1 && !$user) {
                return Response::error('请先注册!');
            }
            //插入短信数据
            $this->mobile = $mobile;
            $this->code = $code;
            $this->type = $type;
            $this->status = self::STA_0;
            if (!$this->save()) {
                return Response::error('短信写入失败!');
            }
            // 发送短信
            $switch = $this->config->getVal('sms_switch') == 1 ? true : false;
            if ($switch) {
                $this->toSend($mobile, $code);
            }
            DB::commit();
            return Response::success(['switch' => $switch, 'code' => $code]);
        } catch (Exception $e) {
            DB::rollBack();
            return Response::error('短信发送失败!');
        }
    }

    /**
     * 发送
     *
     * @param string $mobile    手机号
     * @param string $code      验证码
     * @return
     */
    protected function toSend($mobile, $code)
    {
        try {
            //发送类型
            $type = $this->config->getVal('sms_type');
            switch ($type) {
                case 'aliyun':
                    // 阿里云发送
                    $template = $this->config->getVal('sms_aliyun_template_id');
                    $data = ['template' => $template, 'data' => ['code' => $code]];
                    break;
                case 'qcloud':
                    // 腾讯短信发送
                    $template = $this->config->getVal('sms_qcloud_template_id');
                    $data = ['template' => $template, 'data' => [$code]];
                    break;
                default:
                    throw new ApiException('请先配置短信');
            }
            $easySms = new EasySms($this->sms_config);
            $easySms->send($mobile, $data);
        } catch (\Overtrue\EasySms\Exceptions\NoGatewayAvailableException $e) {
            throw new ApiException('短信发送失败!');
        }
    }

    /**
     * 验证验证码
     *
     * @param  string $mobile
     * @param  string $type
     * @param  string $code
     */
    public function verify($mobile, $type, $code)
    {
        try {
            $log = $this->where(['mobile' => $mobile, 'status' => self::STA_0, 'type' => $type])
                ->orderBy('created_at', 'desc')
                ->first();
            //判断是否发送过（5分钟以内）
            $last_time = 5;
            if (!$log || (time() - $log->created_at->format('U')) > $last_time * 60) {
                return Response::error('请先发送短信!');
            }
            //判断验证码是否一致
            if ($log->code != $code) {
                return Response::error('验证码不正确!');
            }
            return Response::success($log);
        } catch (Exception $e) {
            return Response::error('验证失败!');
        }
    }
}
