<?php

namespace App\Services\User;

use App\Models\User\Tree;
use App\Common\Method\Response;
use App\Services\Luckbox\TermModel;
use App\Services\User\BindLogModel;
use App\Services\User\UsersModel;
use Illuminate\Support\Facades\DB;

/** 用户关系树 */
class TreeModel extends Tree
{
    /**
     * 注册绑定上级
     *
     * @param  object $user
     * @param  array  $param
     */
    public function registerBind($user, $param)
    {
        $bindLogModel = new BindLogModel();
        $parent_id = 0;
        $path = '';
        if ($param['parent_id']) {
            $parent = $this->where('user_id', $param['parent_id'])->first();
            if ($parent) {
                // 上级绑定
                $parent_id = $parent->user_id;
                $path = $parent->path . ',' . $parent->user_id;
                // 绑定记录
                $bindLogModel->user_id = $user->id;
                $bindLogModel->parent_id = $parent->user_id;;
                if (!$bindLogModel->save()) {
                    return Response::error('绑定失败!');
                }
            }
        }
        $treeModel = new self();
        $treeModel->user_id = $user->id;
        $treeModel->parent_id = $parent_id;
        $treeModel->path = trim($path, ',');
        $treeModel->return_times = TermModel::INVITE_USER_TIMES;
        if (!$treeModel->save()) {
            return Response::error('用户关系添加失败!');
        }
        return Response::success();
    }

    /**
     * 是否可以绑定判断
     *
     * @param  object $user
     * @param  object $parent
     */
    public function isBind($user, $parent)
    {
        $tree = $this->where('user_id', $user->id)->first();
        if (!$tree) {
            return Response::error('无法绑定!');
        }
        if ($user->id == $parent->id) {
            return Response::error('无法与自己绑定哦!');
        }
        if ($tree->parent_id != 0) {
            return Response::error('您已绑定过其他人了!');
        }
        if ($tree->parent_id != 0) {
            return Response::error('您已绑定过其他人了!');
        }
        if ($parent->parent_id == $user->id) {
            return Response::error('互相之间不可绑定!');
        }
        return Response::success();
    }

    /**
     * 查找上级N代用户（通过Path)
     *
     * @param string $user_id   用户ID
     * @param integer $level    需要查找的层数(0是不指定层级，取全部上级)
     * @return array
     */
    public function upList($user_id, $level = 0)
    {
        // 上级ID数组
        $arr = [];
        $user = $this->where('user_id', $user_id)->first();
        if ($user && $user['path']) {
            $arr = explode(',', $user['path']);
            // 按上下级排序（最近到最远）
            $arr = array_reverse($arr);
            // 去掉ID为0的数据
            $key =  array_search(0, $arr);
            if ($key) {
                array_splice($arr, array_search(0, $arr), 1);
            }
            // 判断长度
            if ($level > 0 && count($arr) > $level) {
                $arr = array_splice($arr, 0, $level);
            }
        }
        if ($arr) {
            $arr = array_combine(range(1, count($arr)), array_values($arr));
        }
        return $arr;
    }

    /**
     * 查找上级N代用户（通过Parent_id)
     *
     * @param string  $user_id   用户ID
     * @param integer $level    需要查找的层数
     * @param integer $step     查找次数
     * @return array
     */
    public function upLevels($user_id, $level = 3, $step = 1)
    {
        $arr = [];
        //上级用户数据
        $user = $this->where('user_id', $user_id)->first();
        if ($user) {
            $parent = $this->where('user_id', $user->parent_id)->first();
            if ($parent) {
                // 用户ID插入
                $arr = array_merge($arr, [$parent['user_id']]);
                // 判断上级是否有人
                if ($parent['parent_id'] > 0 && $level > $step) {
                    $arr = array_merge($arr, $this->upLevels($parent['user_id'], $level, $step += 1));
                    // 把键变成级别
                    if ($arr) {
                        $arr = array_combine(range(1, count($arr)), array_values($arr));
                    }
                    return $arr;
                }
            }
        }
        return $arr;
    }

    /**
     * 修改用户上级
     * @param object $user     要修改的用户数据
     * @param int    $last_id  被修改的上级ID
     * @return array
     */
    public function editLast($user_id, $last_id = 0)
    {
        // 表名称
        $table = DB::getConfig('prefix') . $this->getTable();
        // 用户数据
        $user = $this->where('user_id', $user_id)->first();
        $parent = $this->where('user_id', $last_id)->first();
        if ($user) {
            // 不能修改为下级
            $list = explode(',', $parent->path ?? '');
            if (in_array($user->id, $list)) {
                return Response::error('不能修改为下级!');
            }
            // 上级ID 和 上级path 路径
            if ($parent) {
                $parent_id = $parent->user_id;
                $new_path = trim($parent->path . ',' . $parent->user_id, ',');
            } else {
                $parent_id = 0;
                $new_path = '';
            }
            $old_path_last = trim($user->path . ',' . $user_id, ',');
            $new_path_last = trim($new_path . ',' . $user_id, ',');
            // 修改用户Path 关系
            DB::update("update $table set path = '$new_path' , parent_id = $parent_id where user_id = $user->user_id");
            DB::update("update $table set path = replace(path,'$old_path_last','$new_path_last')");
            if ($last_id) {
                // 绑定时间
                $bind_time =  time();
                $user->bind_time = $bind_time;
                $userResult = $user->save();
                // 添加绑定记录
                $log = new BindLogModel();
                $log->user_id = $user_id;
                $log->parent_id = $parent_id;
                $logResult = $log->save();
                if (!$userResult || !$logResult) {
                    return Response::error('记录添加失败!');
                }
            } else {
                $user->bind_time = null;
                $userResult = $user->save();
                if (!$userResult) {
                    return Response::error('记录添加失败!');
                }
            }
        }
        return Response::success();
    }

    /**
     * 删除用
     * 
     * @param object $user__id  要删除的用户数据
     * @return array
     */
    public function delUser($user_id)
    {
        // 表名称
        $table = DB::getConfig('prefix') . $this->getTable();
        // 用户数据
        $user = $this->where('user_id', $user_id)->first();
        if ($user) {
            $new_path = $user->path;
            if ($new_path) {
                $new_path .= ',';
            }
            $old_path = trim($user->path . ',' . $user_id, ',') . ',';
            // 修改用户Path 关系
            DB::update("update $table set path = '$user->path', parent_id = $user->parent_id where parent_id = $user->id");
            DB::update("update $table set path = replace(path,'$old_path','$new_path')");
        }
        return Response::success();
    }

    /**
     * 获取当前用户的下N级(指定层级)
     *
     * @param  string  $user_id
     * @param  integer $level
     */
    public function downList($user_id, $level = 1)
    {
        $bindLogModel = new BindLogModel();
        $userModel = new UsersModel();
        $data = [];
        $oneUserList = $bindLogModel->where('parent_id',$user_id)->get()->toArray();
        $oneUserNum = $bindLogModel->where('parent_id',$user_id)->get()->count();
        $oneUserIds = array_column($oneUserList,'user_id');
        $oneUser = $userModel->whereIn('id',$oneUserIds)->get();
        
        $twoUserList = $bindLogModel->whereIn('parent_id',$oneUserIds)->get()->toArray();
        $twoUserNum = $bindLogModel->whereIn('parent_id',$oneUserIds)->get()->count();
        $twoUserIds = array_column($twoUserList,'user_id');
        $twoUser = $userModel->whereIn('id',$twoUserIds)->get();
        
        $threeUserList = $bindLogModel->whereIn('parent_id',$twoUserIds)->get()->toArray();
        $threeUserNum = $bindLogModel->whereIn('parent_id',$twoUserIds)->get()->count();
        $threeUserIds = array_column($threeUserList,'user_id');
        $threeUser = $userModel->whereIn('id',$threeUserIds)->get();
        
        $data = [
            'total_num'=>$oneUserNum + $twoUserNum + $threeUserNum,
            'one_user_list'=>$oneUser,
            'two_user_list'=>$twoUser,
            'three_user_list'=>$threeUser,
            ];
        return $data;
    }
}
