<?php

namespace App\ToolsClass;

use App\Common\Method\Response;

/** 金钱 */
class MoneyTool
{
    /**
     * 红包算法
     *
     * @param string    $money  金额
     * @param integer   $num    剩余数量
     * @param float     $min    最小金额
     * @return array [
     *      $change     此次分出的红包金额
     *      $after      分出后的红包金额
     *      $last_num   红包剩余个数
     *   ]
     */
    public static function getData($money, $num = 1, $min = 0.01)
    {
        // 验证
        if ($num <= 0 || $money <= 0) {
            return Response::error('金额和人数要大于0!');
        }
        $account = $num * $min;
        if (bcsub($account, $money, 2) > 0) {
            return Response::error('红包总金额必须≥' . $account . '元');
        }
        // 剩余一个的时候，直接取剩余红包
        if ($num == 1) {
            $coupon_amount = self::decimal_number($money);
        } else {
            // 剩余的红包的平均金额
            $avg_amount = self::decimal_number($money / $num);
            $coupon_amount = self::calcCouponAmount($avg_amount, $money, $num, $min);
            $coupon_amount = self::decimal_number($coupon_amount);
        }
        $data = [
            'change' => $coupon_amount,
            'after' => bcsub($money, $coupon_amount, 2),
            'last_num' => bcsub($num, 1)
        ];
        return Response::success($data);
    }

    /**
     * 计算分配的红包金额
     *
     * @param float $avg_amount 每次计算的平均金额
     * @param float $amount     剩余可领取金额
     * @param int   $num        剩余可领取的红包个数
     *
     * @return float
     */
    private static function calcCouponAmount($avg_amount, $money, $num, $min)
    {
        // 如果平均金额小于等于最低金额，则直接返回最低金额
        if ($avg_amount <= $min) {
            return $min;
        }
        // 浮动计算
        $coupon_amount = self::decimal_number($avg_amount * (1 + self::apportionRandRatio()));
        // 如果低于最低金额或超过可领取的最大金额，则重新获取
        if (
            $coupon_amount < $min
            || $coupon_amount > self::calcCouponAmountMax($money, $num, $min)
        ) {
            return self::calcCouponAmount($avg_amount, $money, $num, $min);
        }
        return $coupon_amount;
    }

    /**
     * 计算分配的红包金额-可领取的最大金额
     *
     * @param float $amount
     * @param int   $num
     */
    private static function calcCouponAmountMax($amount, $num, $min)
    {
        return $min + $amount - $num * $min;
    }

    /**
     * 红包金额浮动比例
     */
    private static function apportionRandRatio()
    {
        // 60%机率获取剩余平均值的大幅度红包（可能正数、可能负数）
        if (mt_rand(1, 100) <= 60) {
            return mt_rand(-70, 70) / 100; // 上下幅度70%
        }
        return mt_rand(-30, 30) / 100; // 其他情况，上下浮动30%；
    }

    /**
     * 格式化金额，保留2位
     *
     * @param float $amount
     * @return float
     */
    private static function decimal_number($amount)
    {
        if (function_exists('number')) {
            return formatPrice($amount);
        }
        return sprintf('%01.2f', round($amount, 2));
    }
}
